/* thumb2-sha512-asm
 *
 * Copyright (C) 2006-2023 wolfSSL Inc.
 *
 * This file is part of wolfSSL.
 *
 * wolfSSL is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * wolfSSL is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1335, USA
 */

/* Generated using (from wolfssl):
 *   cd ../scripts
 *   ruby ./sha2/sha512.rb thumb2 ../wolfssl/wolfcrypt/src/port/arm/thumb2-sha512-asm.c
 */

#ifdef HAVE_CONFIG_H
    #include <config.h>
#endif /* HAVE_CONFIG_H */
#include <wolfssl/wolfcrypt/settings.h>
#include <wolfssl/wolfcrypt/error-crypt.h>

#ifdef WOLFSSL_ARMASM
#if !defined(__aarch64__) && defined(__thumb__)
#ifdef WOLFSSL_ARMASM_INLINE

#ifdef __IAR_SYSTEMS_ICC__
#define __asm__        asm
#define __volatile__   volatile
#define WOLFSSL_NO_VAR_ASSIGN_REG
#endif /* __IAR_SYSTEMS_ICC__ */
#ifdef __KEIL__
#define __asm__        __asm
#define __volatile__   volatile
#endif /* __KEIL__ */
#ifdef WOLFSSL_SHA512
#include <wolfssl/wolfcrypt/sha512.h>

#ifdef WOLFSSL_ARMASM_NO_NEON
static const uint64_t L_SHA512_transform_len_k[] = {
    0x428a2f98d728ae22UL, 0x7137449123ef65cdUL,
    0xb5c0fbcfec4d3b2fUL, 0xe9b5dba58189dbbcUL,
    0x3956c25bf348b538UL, 0x59f111f1b605d019UL,
    0x923f82a4af194f9bUL, 0xab1c5ed5da6d8118UL,
    0xd807aa98a3030242UL, 0x12835b0145706fbeUL,
    0x243185be4ee4b28cUL, 0x550c7dc3d5ffb4e2UL,
    0x72be5d74f27b896fUL, 0x80deb1fe3b1696b1UL,
    0x9bdc06a725c71235UL, 0xc19bf174cf692694UL,
    0xe49b69c19ef14ad2UL, 0xefbe4786384f25e3UL,
    0x0fc19dc68b8cd5b5UL, 0x240ca1cc77ac9c65UL,
    0x2de92c6f592b0275UL, 0x4a7484aa6ea6e483UL,
    0x5cb0a9dcbd41fbd4UL, 0x76f988da831153b5UL,
    0x983e5152ee66dfabUL, 0xa831c66d2db43210UL,
    0xb00327c898fb213fUL, 0xbf597fc7beef0ee4UL,
    0xc6e00bf33da88fc2UL, 0xd5a79147930aa725UL,
    0x06ca6351e003826fUL, 0x142929670a0e6e70UL,
    0x27b70a8546d22ffcUL, 0x2e1b21385c26c926UL,
    0x4d2c6dfc5ac42aedUL, 0x53380d139d95b3dfUL,
    0x650a73548baf63deUL, 0x766a0abb3c77b2a8UL,
    0x81c2c92e47edaee6UL, 0x92722c851482353bUL,
    0xa2bfe8a14cf10364UL, 0xa81a664bbc423001UL,
    0xc24b8b70d0f89791UL, 0xc76c51a30654be30UL,
    0xd192e819d6ef5218UL, 0xd69906245565a910UL,
    0xf40e35855771202aUL, 0x106aa07032bbd1b8UL,
    0x19a4c116b8d2d0c8UL, 0x1e376c085141ab53UL,
    0x2748774cdf8eeb99UL, 0x34b0bcb5e19b48a8UL,
    0x391c0cb3c5c95a63UL, 0x4ed8aa4ae3418acbUL,
    0x5b9cca4f7763e373UL, 0x682e6ff3d6b2b8a3UL,
    0x748f82ee5defb2fcUL, 0x78a5636f43172f60UL,
    0x84c87814a1f0ab72UL, 0x8cc702081a6439ecUL,
    0x90befffa23631e28UL, 0xa4506cebde82bde9UL,
    0xbef9a3f7b2c67915UL, 0xc67178f2e372532bUL,
    0xca273eceea26619cUL, 0xd186b8c721c0c207UL,
    0xeada7dd6cde0eb1eUL, 0xf57d4f7fee6ed178UL,
    0x06f067aa72176fbaUL, 0x0a637dc5a2c898a6UL,
    0x113f9804bef90daeUL, 0x1b710b35131c471bUL,
    0x28db77f523047d84UL, 0x32caab7b40c72493UL,
    0x3c9ebe0a15c9bebcUL, 0x431d67c49c100d4cUL,
    0x4cc5d4becb3e42b6UL, 0x597f299cfc657e2aUL,
    0x5fcb6fab3ad6faecUL, 0x6c44198c4a475817UL,
};

void Transform_Sha512_Len(wc_Sha512* sha512, const byte* data, word32 len);
#ifndef WOLFSSL_NO_VAR_ASSIGN_REG
void Transform_Sha512_Len(wc_Sha512* sha512_p, const byte* data_p, word32 len_p)
#else
void Transform_Sha512_Len(wc_Sha512* sha512, const byte* data, word32 len)
#endif /* !WOLFSSL_NO_VAR_ASSIGN_REG */
{
#ifndef WOLFSSL_NO_VAR_ASSIGN_REG
    register wc_Sha512* sha512 __asm__ ("r0") = (wc_Sha512*)sha512_p;
    register const byte* data __asm__ ("r1") = (const byte*)data_p;
    register word32 len __asm__ ("r2") = (word32)len_p;
    register uint64_t* L_SHA512_transform_len_k_c __asm__ ("r3") = (uint64_t*)&L_SHA512_transform_len_k;
#endif /* !WOLFSSL_NO_VAR_ASSIGN_REG */

    __asm__ __volatile__ (
        "SUB	sp, sp, #0xc0\n\t"
        "MOV	r3, %[L_SHA512_transform_len_k]\n\t"
        /* Copy digest to add in at end */
        "LDRD	r4, r5, [%[sha512]]\n\t"
        "LDRD	r6, r7, [%[sha512], #8]\n\t"
        "LDRD	r8, r9, [%[sha512], #16]\n\t"
        "LDRD	r10, r11, [%[sha512], #24]\n\t"
        "STRD	r4, r5, [sp, #128]\n\t"
        "STRD	r6, r7, [sp, #136]\n\t"
        "STRD	r8, r9, [sp, #144]\n\t"
        "STRD	r10, r11, [sp, #152]\n\t"
        "LDRD	r4, r5, [%[sha512], #32]\n\t"
        "LDRD	r6, r7, [%[sha512], #40]\n\t"
        "LDRD	r8, r9, [%[sha512], #48]\n\t"
        "LDRD	r10, r11, [%[sha512], #56]\n\t"
        "STRD	r4, r5, [sp, #160]\n\t"
        "STRD	r6, r7, [sp, #168]\n\t"
        "STRD	r8, r9, [sp, #176]\n\t"
        "STRD	r10, r11, [sp, #184]\n\t"
        /* Start of loop processing a block */
        "\n"
    "L_SHA512_transform_len_begin%=:\n\t"
        /* Load, Reverse and Store W */
        "LDR	r4, [%[data]]\n\t"
        "LDR	r5, [%[data], #4]\n\t"
        "LDR	r6, [%[data], #8]\n\t"
        "LDR	r7, [%[data], #12]\n\t"
        "LDR	r8, [%[data], #16]\n\t"
        "LDR	r9, [%[data], #20]\n\t"
        "LDR	r10, [%[data], #24]\n\t"
        "LDR	r11, [%[data], #28]\n\t"
        "REV	r4, r4\n\t"
        "REV	r5, r5\n\t"
        "REV	r6, r6\n\t"
        "REV	r7, r7\n\t"
        "REV	r8, r8\n\t"
        "REV	r9, r9\n\t"
        "REV	r10, r10\n\t"
        "REV	r11, r11\n\t"
        "STR	r5, [sp]\n\t"
        "STR	r4, [sp, #4]\n\t"
        "STR	r7, [sp, #8]\n\t"
        "STR	r6, [sp, #12]\n\t"
        "STR	r9, [sp, #16]\n\t"
        "STR	r8, [sp, #20]\n\t"
        "STR	r11, [sp, #24]\n\t"
        "STR	r10, [sp, #28]\n\t"
        "LDR	r4, [%[data], #32]\n\t"
        "LDR	r5, [%[data], #36]\n\t"
        "LDR	r6, [%[data], #40]\n\t"
        "LDR	r7, [%[data], #44]\n\t"
        "LDR	r8, [%[data], #48]\n\t"
        "LDR	r9, [%[data], #52]\n\t"
        "LDR	r10, [%[data], #56]\n\t"
        "LDR	r11, [%[data], #60]\n\t"
        "REV	r4, r4\n\t"
        "REV	r5, r5\n\t"
        "REV	r6, r6\n\t"
        "REV	r7, r7\n\t"
        "REV	r8, r8\n\t"
        "REV	r9, r9\n\t"
        "REV	r10, r10\n\t"
        "REV	r11, r11\n\t"
        "STR	r5, [sp, #32]\n\t"
        "STR	r4, [sp, #36]\n\t"
        "STR	r7, [sp, #40]\n\t"
        "STR	r6, [sp, #44]\n\t"
        "STR	r9, [sp, #48]\n\t"
        "STR	r8, [sp, #52]\n\t"
        "STR	r11, [sp, #56]\n\t"
        "STR	r10, [sp, #60]\n\t"
        "LDR	r4, [%[data], #64]\n\t"
        "LDR	r5, [%[data], #68]\n\t"
        "LDR	r6, [%[data], #72]\n\t"
        "LDR	r7, [%[data], #76]\n\t"
        "LDR	r8, [%[data], #80]\n\t"
        "LDR	r9, [%[data], #84]\n\t"
        "LDR	r10, [%[data], #88]\n\t"
        "LDR	r11, [%[data], #92]\n\t"
        "REV	r4, r4\n\t"
        "REV	r5, r5\n\t"
        "REV	r6, r6\n\t"
        "REV	r7, r7\n\t"
        "REV	r8, r8\n\t"
        "REV	r9, r9\n\t"
        "REV	r10, r10\n\t"
        "REV	r11, r11\n\t"
        "STR	r5, [sp, #64]\n\t"
        "STR	r4, [sp, #68]\n\t"
        "STR	r7, [sp, #72]\n\t"
        "STR	r6, [sp, #76]\n\t"
        "STR	r9, [sp, #80]\n\t"
        "STR	r8, [sp, #84]\n\t"
        "STR	r11, [sp, #88]\n\t"
        "STR	r10, [sp, #92]\n\t"
        "LDR	r4, [%[data], #96]\n\t"
        "LDR	r5, [%[data], #100]\n\t"
        "LDR	r6, [%[data], #104]\n\t"
        "LDR	r7, [%[data], #108]\n\t"
        "LDR	r8, [%[data], #112]\n\t"
        "LDR	r9, [%[data], #116]\n\t"
        "LDR	r10, [%[data], #120]\n\t"
        "LDR	r11, [%[data], #124]\n\t"
        "REV	r4, r4\n\t"
        "REV	r5, r5\n\t"
        "REV	r6, r6\n\t"
        "REV	r7, r7\n\t"
        "REV	r8, r8\n\t"
        "REV	r9, r9\n\t"
        "REV	r10, r10\n\t"
        "REV	r11, r11\n\t"
        "STR	r5, [sp, #96]\n\t"
        "STR	r4, [sp, #100]\n\t"
        "STR	r7, [sp, #104]\n\t"
        "STR	r6, [sp, #108]\n\t"
        "STR	r9, [sp, #112]\n\t"
        "STR	r8, [sp, #116]\n\t"
        "STR	r11, [sp, #120]\n\t"
        "STR	r10, [sp, #124]\n\t"
        /* Pre-calc: b ^ c */
        "LDRD	r10, r11, [%[sha512], #8]\n\t"
        "LDRD	r4, r5, [%[sha512], #16]\n\t"
        "EOR	r10, r10, r4\n\t"
        "EOR	r11, r11, r5\n\t"
        "MOV	r12, #0x4\n\t"
        /* Start of 16 rounds */
        "\n"
    "L_SHA512_transform_len_start%=:\n\t"
        /* Round 0 */
        "LDRD	r4, r5, [%[sha512], #32]\n\t"
        "LSRS	r6, r4, #14\n\t"
        "LSRS	r7, r5, #14\n\t"
        "ORR	r7, r7, r4, LSL #18\n\t"
        "ORR	r6, r6, r5, LSL #18\n\t"
        "LSRS	r8, r4, #18\n\t"
        "LSRS	r9, r5, #18\n\t"
        "ORR	r9, r9, r4, LSL #14\n\t"
        "ORR	r8, r8, r5, LSL #14\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "LSLS	r8, r4, #23\n\t"
        "LSLS	r9, r5, #23\n\t"
        "ORR	r9, r9, r4, LSR #9\n\t"
        "ORR	r8, r8, r5, LSR #9\n\t"
        "LDRD	r4, r5, [%[sha512], #56]\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "STRD	r4, r5, [%[sha512], #56]\n\t"
        "LDRD	r4, r5, [%[sha512], #32]\n\t"
        "LDRD	r6, r7, [%[sha512], #40]\n\t"
        "LDRD	r8, r9, [%[sha512], #48]\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "AND	r6, r6, r4\n\t"
        "AND	r7, r7, r5\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "LDRD	r4, r5, [%[sha512], #56]\n\t"
        "LDRD	r8, r9, [sp]\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "LDRD	r6, r7, [r3]\n\t"
        "ADDS	r4, r4, r8\n\t"
        "ADC	r5, r5, r9\n\t"
        "LDRD	r8, r9, [%[sha512], #24]\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "STRD	r4, r5, [%[sha512], #56]\n\t"
        "ADDS	r8, r8, r4\n\t"
        "ADC	r9, r9, r5\n\t"
        "LDRD	r4, r5, [%[sha512]]\n\t"
        "STRD	r8, r9, [%[sha512], #24]\n\t"
        "LSRS	r6, r4, #28\n\t"
        "LSRS	r7, r5, #28\n\t"
        "ORR	r7, r7, r4, LSL #4\n\t"
        "ORR	r6, r6, r5, LSL #4\n\t"
        "LSLS	r8, r4, #30\n\t"
        "LSLS	r9, r5, #30\n\t"
        "ORR	r9, r9, r4, LSR #2\n\t"
        "ORR	r8, r8, r5, LSR #2\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "LSLS	r8, r4, #25\n\t"
        "LSLS	r9, r5, #25\n\t"
        "ORR	r9, r9, r4, LSR #7\n\t"
        "ORR	r8, r8, r5, LSR #7\n\t"
        "LDRD	r4, r5, [%[sha512], #56]\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "LDRD	r8, r9, [%[sha512]]\n\t"
        "LDRD	r6, r7, [%[sha512], #8]\n\t"
        "STRD	r4, r5, [%[sha512], #56]\n\t"
        "EOR	r8, r8, r6\n\t"
        "EOR	r9, r9, r7\n\t"
        "AND	r10, r10, r8\n\t"
        "AND	r11, r11, r9\n\t"
        "EOR	r10, r10, r6\n\t"
        "EOR	r11, r11, r7\n\t"
        "LDRD	r6, r7, [%[sha512], #56]\n\t"
        "ADDS	r6, r6, r10\n\t"
        "ADC	r7, r7, r11\n\t"
        "STRD	r6, r7, [%[sha512], #56]\n\t"
        "MOV	r10, r8\n\t"
        "MOV	r11, r9\n\t"
        /* Calc new W[0] */
        "LDRD	r4, r5, [sp, #112]\n\t"
        "LSRS	r6, r4, #19\n\t"
        "LSRS	r7, r5, #19\n\t"
        "ORR	r7, r7, r4, LSL #13\n\t"
        "ORR	r6, r6, r5, LSL #13\n\t"
        "LSLS	r8, r4, #3\n\t"
        "LSLS	r9, r5, #3\n\t"
        "ORR	r9, r9, r4, LSR #29\n\t"
        "ORR	r8, r8, r5, LSR #29\n\t"
        "EOR	r7, r7, r9\n\t"
        "EOR	r6, r6, r8\n\t"
        "LSRS	r8, r4, #6\n\t"
        "LSRS	r9, r5, #6\n\t"
        "ORR	r8, r8, r5, LSL #26\n\t"
        "EOR	r7, r7, r9\n\t"
        "EOR	r6, r6, r8\n\t"
        "LDRD	r4, r5, [sp]\n\t"
        "LDRD	r8, r9, [sp, #72]\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "ADDS	r4, r4, r8\n\t"
        "ADC	r5, r5, r9\n\t"
        "STRD	r4, r5, [sp]\n\t"
        "LDRD	r4, r5, [sp, #8]\n\t"
        "LSRS	r6, r4, #1\n\t"
        "LSRS	r7, r5, #1\n\t"
        "ORR	r7, r7, r4, LSL #31\n\t"
        "ORR	r6, r6, r5, LSL #31\n\t"
        "LSRS	r8, r4, #8\n\t"
        "LSRS	r9, r5, #8\n\t"
        "ORR	r9, r9, r4, LSL #24\n\t"
        "ORR	r8, r8, r5, LSL #24\n\t"
        "EOR	r7, r7, r9\n\t"
        "EOR	r6, r6, r8\n\t"
        "LSRS	r8, r4, #7\n\t"
        "LSRS	r9, r5, #7\n\t"
        "ORR	r8, r8, r5, LSL #25\n\t"
        "EOR	r7, r7, r9\n\t"
        "EOR	r6, r6, r8\n\t"
        "LDRD	r4, r5, [sp]\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "STRD	r4, r5, [sp]\n\t"
        /* Round 1 */
        "LDRD	r4, r5, [%[sha512], #24]\n\t"
        "LSRS	r6, r4, #14\n\t"
        "LSRS	r7, r5, #14\n\t"
        "ORR	r7, r7, r4, LSL #18\n\t"
        "ORR	r6, r6, r5, LSL #18\n\t"
        "LSRS	r8, r4, #18\n\t"
        "LSRS	r9, r5, #18\n\t"
        "ORR	r9, r9, r4, LSL #14\n\t"
        "ORR	r8, r8, r5, LSL #14\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "LSLS	r8, r4, #23\n\t"
        "LSLS	r9, r5, #23\n\t"
        "ORR	r9, r9, r4, LSR #9\n\t"
        "ORR	r8, r8, r5, LSR #9\n\t"
        "LDRD	r4, r5, [%[sha512], #48]\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "STRD	r4, r5, [%[sha512], #48]\n\t"
        "LDRD	r4, r5, [%[sha512], #24]\n\t"
        "LDRD	r6, r7, [%[sha512], #32]\n\t"
        "LDRD	r8, r9, [%[sha512], #40]\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "AND	r6, r6, r4\n\t"
        "AND	r7, r7, r5\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "LDRD	r4, r5, [%[sha512], #48]\n\t"
        "LDRD	r8, r9, [sp, #8]\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "LDRD	r6, r7, [r3, #8]\n\t"
        "ADDS	r4, r4, r8\n\t"
        "ADC	r5, r5, r9\n\t"
        "LDRD	r8, r9, [%[sha512], #16]\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "STRD	r4, r5, [%[sha512], #48]\n\t"
        "ADDS	r8, r8, r4\n\t"
        "ADC	r9, r9, r5\n\t"
        "LDRD	r4, r5, [%[sha512], #56]\n\t"
        "STRD	r8, r9, [%[sha512], #16]\n\t"
        "LSRS	r6, r4, #28\n\t"
        "LSRS	r7, r5, #28\n\t"
        "ORR	r7, r7, r4, LSL #4\n\t"
        "ORR	r6, r6, r5, LSL #4\n\t"
        "LSLS	r8, r4, #30\n\t"
        "LSLS	r9, r5, #30\n\t"
        "ORR	r9, r9, r4, LSR #2\n\t"
        "ORR	r8, r8, r5, LSR #2\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "LSLS	r8, r4, #25\n\t"
        "LSLS	r9, r5, #25\n\t"
        "ORR	r9, r9, r4, LSR #7\n\t"
        "ORR	r8, r8, r5, LSR #7\n\t"
        "LDRD	r4, r5, [%[sha512], #48]\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "LDRD	r8, r9, [%[sha512], #56]\n\t"
        "LDRD	r6, r7, [%[sha512]]\n\t"
        "STRD	r4, r5, [%[sha512], #48]\n\t"
        "EOR	r8, r8, r6\n\t"
        "EOR	r9, r9, r7\n\t"
        "AND	r10, r10, r8\n\t"
        "AND	r11, r11, r9\n\t"
        "EOR	r10, r10, r6\n\t"
        "EOR	r11, r11, r7\n\t"
        "LDRD	r6, r7, [%[sha512], #48]\n\t"
        "ADDS	r6, r6, r10\n\t"
        "ADC	r7, r7, r11\n\t"
        "STRD	r6, r7, [%[sha512], #48]\n\t"
        "MOV	r10, r8\n\t"
        "MOV	r11, r9\n\t"
        /* Calc new W[1] */
        "LDRD	r4, r5, [sp, #120]\n\t"
        "LSRS	r6, r4, #19\n\t"
        "LSRS	r7, r5, #19\n\t"
        "ORR	r7, r7, r4, LSL #13\n\t"
        "ORR	r6, r6, r5, LSL #13\n\t"
        "LSLS	r8, r4, #3\n\t"
        "LSLS	r9, r5, #3\n\t"
        "ORR	r9, r9, r4, LSR #29\n\t"
        "ORR	r8, r8, r5, LSR #29\n\t"
        "EOR	r7, r7, r9\n\t"
        "EOR	r6, r6, r8\n\t"
        "LSRS	r8, r4, #6\n\t"
        "LSRS	r9, r5, #6\n\t"
        "ORR	r8, r8, r5, LSL #26\n\t"
        "EOR	r7, r7, r9\n\t"
        "EOR	r6, r6, r8\n\t"
        "LDRD	r4, r5, [sp, #8]\n\t"
        "LDRD	r8, r9, [sp, #80]\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "ADDS	r4, r4, r8\n\t"
        "ADC	r5, r5, r9\n\t"
        "STRD	r4, r5, [sp, #8]\n\t"
        "LDRD	r4, r5, [sp, #16]\n\t"
        "LSRS	r6, r4, #1\n\t"
        "LSRS	r7, r5, #1\n\t"
        "ORR	r7, r7, r4, LSL #31\n\t"
        "ORR	r6, r6, r5, LSL #31\n\t"
        "LSRS	r8, r4, #8\n\t"
        "LSRS	r9, r5, #8\n\t"
        "ORR	r9, r9, r4, LSL #24\n\t"
        "ORR	r8, r8, r5, LSL #24\n\t"
        "EOR	r7, r7, r9\n\t"
        "EOR	r6, r6, r8\n\t"
        "LSRS	r8, r4, #7\n\t"
        "LSRS	r9, r5, #7\n\t"
        "ORR	r8, r8, r5, LSL #25\n\t"
        "EOR	r7, r7, r9\n\t"
        "EOR	r6, r6, r8\n\t"
        "LDRD	r4, r5, [sp, #8]\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "STRD	r4, r5, [sp, #8]\n\t"
        /* Round 2 */
        "LDRD	r4, r5, [%[sha512], #16]\n\t"
        "LSRS	r6, r4, #14\n\t"
        "LSRS	r7, r5, #14\n\t"
        "ORR	r7, r7, r4, LSL #18\n\t"
        "ORR	r6, r6, r5, LSL #18\n\t"
        "LSRS	r8, r4, #18\n\t"
        "LSRS	r9, r5, #18\n\t"
        "ORR	r9, r9, r4, LSL #14\n\t"
        "ORR	r8, r8, r5, LSL #14\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "LSLS	r8, r4, #23\n\t"
        "LSLS	r9, r5, #23\n\t"
        "ORR	r9, r9, r4, LSR #9\n\t"
        "ORR	r8, r8, r5, LSR #9\n\t"
        "LDRD	r4, r5, [%[sha512], #40]\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "STRD	r4, r5, [%[sha512], #40]\n\t"
        "LDRD	r4, r5, [%[sha512], #16]\n\t"
        "LDRD	r6, r7, [%[sha512], #24]\n\t"
        "LDRD	r8, r9, [%[sha512], #32]\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "AND	r6, r6, r4\n\t"
        "AND	r7, r7, r5\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "LDRD	r4, r5, [%[sha512], #40]\n\t"
        "LDRD	r8, r9, [sp, #16]\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "LDRD	r6, r7, [r3, #16]\n\t"
        "ADDS	r4, r4, r8\n\t"
        "ADC	r5, r5, r9\n\t"
        "LDRD	r8, r9, [%[sha512], #8]\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "STRD	r4, r5, [%[sha512], #40]\n\t"
        "ADDS	r8, r8, r4\n\t"
        "ADC	r9, r9, r5\n\t"
        "LDRD	r4, r5, [%[sha512], #48]\n\t"
        "STRD	r8, r9, [%[sha512], #8]\n\t"
        "LSRS	r6, r4, #28\n\t"
        "LSRS	r7, r5, #28\n\t"
        "ORR	r7, r7, r4, LSL #4\n\t"
        "ORR	r6, r6, r5, LSL #4\n\t"
        "LSLS	r8, r4, #30\n\t"
        "LSLS	r9, r5, #30\n\t"
        "ORR	r9, r9, r4, LSR #2\n\t"
        "ORR	r8, r8, r5, LSR #2\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "LSLS	r8, r4, #25\n\t"
        "LSLS	r9, r5, #25\n\t"
        "ORR	r9, r9, r4, LSR #7\n\t"
        "ORR	r8, r8, r5, LSR #7\n\t"
        "LDRD	r4, r5, [%[sha512], #40]\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "LDRD	r8, r9, [%[sha512], #48]\n\t"
        "LDRD	r6, r7, [%[sha512], #56]\n\t"
        "STRD	r4, r5, [%[sha512], #40]\n\t"
        "EOR	r8, r8, r6\n\t"
        "EOR	r9, r9, r7\n\t"
        "AND	r10, r10, r8\n\t"
        "AND	r11, r11, r9\n\t"
        "EOR	r10, r10, r6\n\t"
        "EOR	r11, r11, r7\n\t"
        "LDRD	r6, r7, [%[sha512], #40]\n\t"
        "ADDS	r6, r6, r10\n\t"
        "ADC	r7, r7, r11\n\t"
        "STRD	r6, r7, [%[sha512], #40]\n\t"
        "MOV	r10, r8\n\t"
        "MOV	r11, r9\n\t"
        /* Calc new W[2] */
        "LDRD	r4, r5, [sp]\n\t"
        "LSRS	r6, r4, #19\n\t"
        "LSRS	r7, r5, #19\n\t"
        "ORR	r7, r7, r4, LSL #13\n\t"
        "ORR	r6, r6, r5, LSL #13\n\t"
        "LSLS	r8, r4, #3\n\t"
        "LSLS	r9, r5, #3\n\t"
        "ORR	r9, r9, r4, LSR #29\n\t"
        "ORR	r8, r8, r5, LSR #29\n\t"
        "EOR	r7, r7, r9\n\t"
        "EOR	r6, r6, r8\n\t"
        "LSRS	r8, r4, #6\n\t"
        "LSRS	r9, r5, #6\n\t"
        "ORR	r8, r8, r5, LSL #26\n\t"
        "EOR	r7, r7, r9\n\t"
        "EOR	r6, r6, r8\n\t"
        "LDRD	r4, r5, [sp, #16]\n\t"
        "LDRD	r8, r9, [sp, #88]\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "ADDS	r4, r4, r8\n\t"
        "ADC	r5, r5, r9\n\t"
        "STRD	r4, r5, [sp, #16]\n\t"
        "LDRD	r4, r5, [sp, #24]\n\t"
        "LSRS	r6, r4, #1\n\t"
        "LSRS	r7, r5, #1\n\t"
        "ORR	r7, r7, r4, LSL #31\n\t"
        "ORR	r6, r6, r5, LSL #31\n\t"
        "LSRS	r8, r4, #8\n\t"
        "LSRS	r9, r5, #8\n\t"
        "ORR	r9, r9, r4, LSL #24\n\t"
        "ORR	r8, r8, r5, LSL #24\n\t"
        "EOR	r7, r7, r9\n\t"
        "EOR	r6, r6, r8\n\t"
        "LSRS	r8, r4, #7\n\t"
        "LSRS	r9, r5, #7\n\t"
        "ORR	r8, r8, r5, LSL #25\n\t"
        "EOR	r7, r7, r9\n\t"
        "EOR	r6, r6, r8\n\t"
        "LDRD	r4, r5, [sp, #16]\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "STRD	r4, r5, [sp, #16]\n\t"
        /* Round 3 */
        "LDRD	r4, r5, [%[sha512], #8]\n\t"
        "LSRS	r6, r4, #14\n\t"
        "LSRS	r7, r5, #14\n\t"
        "ORR	r7, r7, r4, LSL #18\n\t"
        "ORR	r6, r6, r5, LSL #18\n\t"
        "LSRS	r8, r4, #18\n\t"
        "LSRS	r9, r5, #18\n\t"
        "ORR	r9, r9, r4, LSL #14\n\t"
        "ORR	r8, r8, r5, LSL #14\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "LSLS	r8, r4, #23\n\t"
        "LSLS	r9, r5, #23\n\t"
        "ORR	r9, r9, r4, LSR #9\n\t"
        "ORR	r8, r8, r5, LSR #9\n\t"
        "LDRD	r4, r5, [%[sha512], #32]\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "STRD	r4, r5, [%[sha512], #32]\n\t"
        "LDRD	r4, r5, [%[sha512], #8]\n\t"
        "LDRD	r6, r7, [%[sha512], #16]\n\t"
        "LDRD	r8, r9, [%[sha512], #24]\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "AND	r6, r6, r4\n\t"
        "AND	r7, r7, r5\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "LDRD	r4, r5, [%[sha512], #32]\n\t"
        "LDRD	r8, r9, [sp, #24]\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "LDRD	r6, r7, [r3, #24]\n\t"
        "ADDS	r4, r4, r8\n\t"
        "ADC	r5, r5, r9\n\t"
        "LDRD	r8, r9, [%[sha512]]\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "STRD	r4, r5, [%[sha512], #32]\n\t"
        "ADDS	r8, r8, r4\n\t"
        "ADC	r9, r9, r5\n\t"
        "LDRD	r4, r5, [%[sha512], #40]\n\t"
        "STRD	r8, r9, [%[sha512]]\n\t"
        "LSRS	r6, r4, #28\n\t"
        "LSRS	r7, r5, #28\n\t"
        "ORR	r7, r7, r4, LSL #4\n\t"
        "ORR	r6, r6, r5, LSL #4\n\t"
        "LSLS	r8, r4, #30\n\t"
        "LSLS	r9, r5, #30\n\t"
        "ORR	r9, r9, r4, LSR #2\n\t"
        "ORR	r8, r8, r5, LSR #2\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "LSLS	r8, r4, #25\n\t"
        "LSLS	r9, r5, #25\n\t"
        "ORR	r9, r9, r4, LSR #7\n\t"
        "ORR	r8, r8, r5, LSR #7\n\t"
        "LDRD	r4, r5, [%[sha512], #32]\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "LDRD	r8, r9, [%[sha512], #40]\n\t"
        "LDRD	r6, r7, [%[sha512], #48]\n\t"
        "STRD	r4, r5, [%[sha512], #32]\n\t"
        "EOR	r8, r8, r6\n\t"
        "EOR	r9, r9, r7\n\t"
        "AND	r10, r10, r8\n\t"
        "AND	r11, r11, r9\n\t"
        "EOR	r10, r10, r6\n\t"
        "EOR	r11, r11, r7\n\t"
        "LDRD	r6, r7, [%[sha512], #32]\n\t"
        "ADDS	r6, r6, r10\n\t"
        "ADC	r7, r7, r11\n\t"
        "STRD	r6, r7, [%[sha512], #32]\n\t"
        "MOV	r10, r8\n\t"
        "MOV	r11, r9\n\t"
        /* Calc new W[3] */
        "LDRD	r4, r5, [sp, #8]\n\t"
        "LSRS	r6, r4, #19\n\t"
        "LSRS	r7, r5, #19\n\t"
        "ORR	r7, r7, r4, LSL #13\n\t"
        "ORR	r6, r6, r5, LSL #13\n\t"
        "LSLS	r8, r4, #3\n\t"
        "LSLS	r9, r5, #3\n\t"
        "ORR	r9, r9, r4, LSR #29\n\t"
        "ORR	r8, r8, r5, LSR #29\n\t"
        "EOR	r7, r7, r9\n\t"
        "EOR	r6, r6, r8\n\t"
        "LSRS	r8, r4, #6\n\t"
        "LSRS	r9, r5, #6\n\t"
        "ORR	r8, r8, r5, LSL #26\n\t"
        "EOR	r7, r7, r9\n\t"
        "EOR	r6, r6, r8\n\t"
        "LDRD	r4, r5, [sp, #24]\n\t"
        "LDRD	r8, r9, [sp, #96]\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "ADDS	r4, r4, r8\n\t"
        "ADC	r5, r5, r9\n\t"
        "STRD	r4, r5, [sp, #24]\n\t"
        "LDRD	r4, r5, [sp, #32]\n\t"
        "LSRS	r6, r4, #1\n\t"
        "LSRS	r7, r5, #1\n\t"
        "ORR	r7, r7, r4, LSL #31\n\t"
        "ORR	r6, r6, r5, LSL #31\n\t"
        "LSRS	r8, r4, #8\n\t"
        "LSRS	r9, r5, #8\n\t"
        "ORR	r9, r9, r4, LSL #24\n\t"
        "ORR	r8, r8, r5, LSL #24\n\t"
        "EOR	r7, r7, r9\n\t"
        "EOR	r6, r6, r8\n\t"
        "LSRS	r8, r4, #7\n\t"
        "LSRS	r9, r5, #7\n\t"
        "ORR	r8, r8, r5, LSL #25\n\t"
        "EOR	r7, r7, r9\n\t"
        "EOR	r6, r6, r8\n\t"
        "LDRD	r4, r5, [sp, #24]\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "STRD	r4, r5, [sp, #24]\n\t"
        /* Round 4 */
        "LDRD	r4, r5, [%[sha512]]\n\t"
        "LSRS	r6, r4, #14\n\t"
        "LSRS	r7, r5, #14\n\t"
        "ORR	r7, r7, r4, LSL #18\n\t"
        "ORR	r6, r6, r5, LSL #18\n\t"
        "LSRS	r8, r4, #18\n\t"
        "LSRS	r9, r5, #18\n\t"
        "ORR	r9, r9, r4, LSL #14\n\t"
        "ORR	r8, r8, r5, LSL #14\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "LSLS	r8, r4, #23\n\t"
        "LSLS	r9, r5, #23\n\t"
        "ORR	r9, r9, r4, LSR #9\n\t"
        "ORR	r8, r8, r5, LSR #9\n\t"
        "LDRD	r4, r5, [%[sha512], #24]\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "STRD	r4, r5, [%[sha512], #24]\n\t"
        "LDRD	r4, r5, [%[sha512]]\n\t"
        "LDRD	r6, r7, [%[sha512], #8]\n\t"
        "LDRD	r8, r9, [%[sha512], #16]\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "AND	r6, r6, r4\n\t"
        "AND	r7, r7, r5\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "LDRD	r4, r5, [%[sha512], #24]\n\t"
        "LDRD	r8, r9, [sp, #32]\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "LDRD	r6, r7, [r3, #32]\n\t"
        "ADDS	r4, r4, r8\n\t"
        "ADC	r5, r5, r9\n\t"
        "LDRD	r8, r9, [%[sha512], #56]\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "STRD	r4, r5, [%[sha512], #24]\n\t"
        "ADDS	r8, r8, r4\n\t"
        "ADC	r9, r9, r5\n\t"
        "LDRD	r4, r5, [%[sha512], #32]\n\t"
        "STRD	r8, r9, [%[sha512], #56]\n\t"
        "LSRS	r6, r4, #28\n\t"
        "LSRS	r7, r5, #28\n\t"
        "ORR	r7, r7, r4, LSL #4\n\t"
        "ORR	r6, r6, r5, LSL #4\n\t"
        "LSLS	r8, r4, #30\n\t"
        "LSLS	r9, r5, #30\n\t"
        "ORR	r9, r9, r4, LSR #2\n\t"
        "ORR	r8, r8, r5, LSR #2\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "LSLS	r8, r4, #25\n\t"
        "LSLS	r9, r5, #25\n\t"
        "ORR	r9, r9, r4, LSR #7\n\t"
        "ORR	r8, r8, r5, LSR #7\n\t"
        "LDRD	r4, r5, [%[sha512], #24]\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "LDRD	r8, r9, [%[sha512], #32]\n\t"
        "LDRD	r6, r7, [%[sha512], #40]\n\t"
        "STRD	r4, r5, [%[sha512], #24]\n\t"
        "EOR	r8, r8, r6\n\t"
        "EOR	r9, r9, r7\n\t"
        "AND	r10, r10, r8\n\t"
        "AND	r11, r11, r9\n\t"
        "EOR	r10, r10, r6\n\t"
        "EOR	r11, r11, r7\n\t"
        "LDRD	r6, r7, [%[sha512], #24]\n\t"
        "ADDS	r6, r6, r10\n\t"
        "ADC	r7, r7, r11\n\t"
        "STRD	r6, r7, [%[sha512], #24]\n\t"
        "MOV	r10, r8\n\t"
        "MOV	r11, r9\n\t"
        /* Calc new W[4] */
        "LDRD	r4, r5, [sp, #16]\n\t"
        "LSRS	r6, r4, #19\n\t"
        "LSRS	r7, r5, #19\n\t"
        "ORR	r7, r7, r4, LSL #13\n\t"
        "ORR	r6, r6, r5, LSL #13\n\t"
        "LSLS	r8, r4, #3\n\t"
        "LSLS	r9, r5, #3\n\t"
        "ORR	r9, r9, r4, LSR #29\n\t"
        "ORR	r8, r8, r5, LSR #29\n\t"
        "EOR	r7, r7, r9\n\t"
        "EOR	r6, r6, r8\n\t"
        "LSRS	r8, r4, #6\n\t"
        "LSRS	r9, r5, #6\n\t"
        "ORR	r8, r8, r5, LSL #26\n\t"
        "EOR	r7, r7, r9\n\t"
        "EOR	r6, r6, r8\n\t"
        "LDRD	r4, r5, [sp, #32]\n\t"
        "LDRD	r8, r9, [sp, #104]\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "ADDS	r4, r4, r8\n\t"
        "ADC	r5, r5, r9\n\t"
        "STRD	r4, r5, [sp, #32]\n\t"
        "LDRD	r4, r5, [sp, #40]\n\t"
        "LSRS	r6, r4, #1\n\t"
        "LSRS	r7, r5, #1\n\t"
        "ORR	r7, r7, r4, LSL #31\n\t"
        "ORR	r6, r6, r5, LSL #31\n\t"
        "LSRS	r8, r4, #8\n\t"
        "LSRS	r9, r5, #8\n\t"
        "ORR	r9, r9, r4, LSL #24\n\t"
        "ORR	r8, r8, r5, LSL #24\n\t"
        "EOR	r7, r7, r9\n\t"
        "EOR	r6, r6, r8\n\t"
        "LSRS	r8, r4, #7\n\t"
        "LSRS	r9, r5, #7\n\t"
        "ORR	r8, r8, r5, LSL #25\n\t"
        "EOR	r7, r7, r9\n\t"
        "EOR	r6, r6, r8\n\t"
        "LDRD	r4, r5, [sp, #32]\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "STRD	r4, r5, [sp, #32]\n\t"
        /* Round 5 */
        "LDRD	r4, r5, [%[sha512], #56]\n\t"
        "LSRS	r6, r4, #14\n\t"
        "LSRS	r7, r5, #14\n\t"
        "ORR	r7, r7, r4, LSL #18\n\t"
        "ORR	r6, r6, r5, LSL #18\n\t"
        "LSRS	r8, r4, #18\n\t"
        "LSRS	r9, r5, #18\n\t"
        "ORR	r9, r9, r4, LSL #14\n\t"
        "ORR	r8, r8, r5, LSL #14\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "LSLS	r8, r4, #23\n\t"
        "LSLS	r9, r5, #23\n\t"
        "ORR	r9, r9, r4, LSR #9\n\t"
        "ORR	r8, r8, r5, LSR #9\n\t"
        "LDRD	r4, r5, [%[sha512], #16]\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "STRD	r4, r5, [%[sha512], #16]\n\t"
        "LDRD	r4, r5, [%[sha512], #56]\n\t"
        "LDRD	r6, r7, [%[sha512]]\n\t"
        "LDRD	r8, r9, [%[sha512], #8]\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "AND	r6, r6, r4\n\t"
        "AND	r7, r7, r5\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "LDRD	r4, r5, [%[sha512], #16]\n\t"
        "LDRD	r8, r9, [sp, #40]\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "LDRD	r6, r7, [r3, #40]\n\t"
        "ADDS	r4, r4, r8\n\t"
        "ADC	r5, r5, r9\n\t"
        "LDRD	r8, r9, [%[sha512], #48]\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "STRD	r4, r5, [%[sha512], #16]\n\t"
        "ADDS	r8, r8, r4\n\t"
        "ADC	r9, r9, r5\n\t"
        "LDRD	r4, r5, [%[sha512], #24]\n\t"
        "STRD	r8, r9, [%[sha512], #48]\n\t"
        "LSRS	r6, r4, #28\n\t"
        "LSRS	r7, r5, #28\n\t"
        "ORR	r7, r7, r4, LSL #4\n\t"
        "ORR	r6, r6, r5, LSL #4\n\t"
        "LSLS	r8, r4, #30\n\t"
        "LSLS	r9, r5, #30\n\t"
        "ORR	r9, r9, r4, LSR #2\n\t"
        "ORR	r8, r8, r5, LSR #2\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "LSLS	r8, r4, #25\n\t"
        "LSLS	r9, r5, #25\n\t"
        "ORR	r9, r9, r4, LSR #7\n\t"
        "ORR	r8, r8, r5, LSR #7\n\t"
        "LDRD	r4, r5, [%[sha512], #16]\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "LDRD	r8, r9, [%[sha512], #24]\n\t"
        "LDRD	r6, r7, [%[sha512], #32]\n\t"
        "STRD	r4, r5, [%[sha512], #16]\n\t"
        "EOR	r8, r8, r6\n\t"
        "EOR	r9, r9, r7\n\t"
        "AND	r10, r10, r8\n\t"
        "AND	r11, r11, r9\n\t"
        "EOR	r10, r10, r6\n\t"
        "EOR	r11, r11, r7\n\t"
        "LDRD	r6, r7, [%[sha512], #16]\n\t"
        "ADDS	r6, r6, r10\n\t"
        "ADC	r7, r7, r11\n\t"
        "STRD	r6, r7, [%[sha512], #16]\n\t"
        "MOV	r10, r8\n\t"
        "MOV	r11, r9\n\t"
        /* Calc new W[5] */
        "LDRD	r4, r5, [sp, #24]\n\t"
        "LSRS	r6, r4, #19\n\t"
        "LSRS	r7, r5, #19\n\t"
        "ORR	r7, r7, r4, LSL #13\n\t"
        "ORR	r6, r6, r5, LSL #13\n\t"
        "LSLS	r8, r4, #3\n\t"
        "LSLS	r9, r5, #3\n\t"
        "ORR	r9, r9, r4, LSR #29\n\t"
        "ORR	r8, r8, r5, LSR #29\n\t"
        "EOR	r7, r7, r9\n\t"
        "EOR	r6, r6, r8\n\t"
        "LSRS	r8, r4, #6\n\t"
        "LSRS	r9, r5, #6\n\t"
        "ORR	r8, r8, r5, LSL #26\n\t"
        "EOR	r7, r7, r9\n\t"
        "EOR	r6, r6, r8\n\t"
        "LDRD	r4, r5, [sp, #40]\n\t"
        "LDRD	r8, r9, [sp, #112]\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "ADDS	r4, r4, r8\n\t"
        "ADC	r5, r5, r9\n\t"
        "STRD	r4, r5, [sp, #40]\n\t"
        "LDRD	r4, r5, [sp, #48]\n\t"
        "LSRS	r6, r4, #1\n\t"
        "LSRS	r7, r5, #1\n\t"
        "ORR	r7, r7, r4, LSL #31\n\t"
        "ORR	r6, r6, r5, LSL #31\n\t"
        "LSRS	r8, r4, #8\n\t"
        "LSRS	r9, r5, #8\n\t"
        "ORR	r9, r9, r4, LSL #24\n\t"
        "ORR	r8, r8, r5, LSL #24\n\t"
        "EOR	r7, r7, r9\n\t"
        "EOR	r6, r6, r8\n\t"
        "LSRS	r8, r4, #7\n\t"
        "LSRS	r9, r5, #7\n\t"
        "ORR	r8, r8, r5, LSL #25\n\t"
        "EOR	r7, r7, r9\n\t"
        "EOR	r6, r6, r8\n\t"
        "LDRD	r4, r5, [sp, #40]\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "STRD	r4, r5, [sp, #40]\n\t"
        /* Round 6 */
        "LDRD	r4, r5, [%[sha512], #48]\n\t"
        "LSRS	r6, r4, #14\n\t"
        "LSRS	r7, r5, #14\n\t"
        "ORR	r7, r7, r4, LSL #18\n\t"
        "ORR	r6, r6, r5, LSL #18\n\t"
        "LSRS	r8, r4, #18\n\t"
        "LSRS	r9, r5, #18\n\t"
        "ORR	r9, r9, r4, LSL #14\n\t"
        "ORR	r8, r8, r5, LSL #14\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "LSLS	r8, r4, #23\n\t"
        "LSLS	r9, r5, #23\n\t"
        "ORR	r9, r9, r4, LSR #9\n\t"
        "ORR	r8, r8, r5, LSR #9\n\t"
        "LDRD	r4, r5, [%[sha512], #8]\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "STRD	r4, r5, [%[sha512], #8]\n\t"
        "LDRD	r4, r5, [%[sha512], #48]\n\t"
        "LDRD	r6, r7, [%[sha512], #56]\n\t"
        "LDRD	r8, r9, [%[sha512]]\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "AND	r6, r6, r4\n\t"
        "AND	r7, r7, r5\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "LDRD	r4, r5, [%[sha512], #8]\n\t"
        "LDRD	r8, r9, [sp, #48]\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "LDRD	r6, r7, [r3, #48]\n\t"
        "ADDS	r4, r4, r8\n\t"
        "ADC	r5, r5, r9\n\t"
        "LDRD	r8, r9, [%[sha512], #40]\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "STRD	r4, r5, [%[sha512], #8]\n\t"
        "ADDS	r8, r8, r4\n\t"
        "ADC	r9, r9, r5\n\t"
        "LDRD	r4, r5, [%[sha512], #16]\n\t"
        "STRD	r8, r9, [%[sha512], #40]\n\t"
        "LSRS	r6, r4, #28\n\t"
        "LSRS	r7, r5, #28\n\t"
        "ORR	r7, r7, r4, LSL #4\n\t"
        "ORR	r6, r6, r5, LSL #4\n\t"
        "LSLS	r8, r4, #30\n\t"
        "LSLS	r9, r5, #30\n\t"
        "ORR	r9, r9, r4, LSR #2\n\t"
        "ORR	r8, r8, r5, LSR #2\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "LSLS	r8, r4, #25\n\t"
        "LSLS	r9, r5, #25\n\t"
        "ORR	r9, r9, r4, LSR #7\n\t"
        "ORR	r8, r8, r5, LSR #7\n\t"
        "LDRD	r4, r5, [%[sha512], #8]\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "LDRD	r8, r9, [%[sha512], #16]\n\t"
        "LDRD	r6, r7, [%[sha512], #24]\n\t"
        "STRD	r4, r5, [%[sha512], #8]\n\t"
        "EOR	r8, r8, r6\n\t"
        "EOR	r9, r9, r7\n\t"
        "AND	r10, r10, r8\n\t"
        "AND	r11, r11, r9\n\t"
        "EOR	r10, r10, r6\n\t"
        "EOR	r11, r11, r7\n\t"
        "LDRD	r6, r7, [%[sha512], #8]\n\t"
        "ADDS	r6, r6, r10\n\t"
        "ADC	r7, r7, r11\n\t"
        "STRD	r6, r7, [%[sha512], #8]\n\t"
        "MOV	r10, r8\n\t"
        "MOV	r11, r9\n\t"
        /* Calc new W[6] */
        "LDRD	r4, r5, [sp, #32]\n\t"
        "LSRS	r6, r4, #19\n\t"
        "LSRS	r7, r5, #19\n\t"
        "ORR	r7, r7, r4, LSL #13\n\t"
        "ORR	r6, r6, r5, LSL #13\n\t"
        "LSLS	r8, r4, #3\n\t"
        "LSLS	r9, r5, #3\n\t"
        "ORR	r9, r9, r4, LSR #29\n\t"
        "ORR	r8, r8, r5, LSR #29\n\t"
        "EOR	r7, r7, r9\n\t"
        "EOR	r6, r6, r8\n\t"
        "LSRS	r8, r4, #6\n\t"
        "LSRS	r9, r5, #6\n\t"
        "ORR	r8, r8, r5, LSL #26\n\t"
        "EOR	r7, r7, r9\n\t"
        "EOR	r6, r6, r8\n\t"
        "LDRD	r4, r5, [sp, #48]\n\t"
        "LDRD	r8, r9, [sp, #120]\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "ADDS	r4, r4, r8\n\t"
        "ADC	r5, r5, r9\n\t"
        "STRD	r4, r5, [sp, #48]\n\t"
        "LDRD	r4, r5, [sp, #56]\n\t"
        "LSRS	r6, r4, #1\n\t"
        "LSRS	r7, r5, #1\n\t"
        "ORR	r7, r7, r4, LSL #31\n\t"
        "ORR	r6, r6, r5, LSL #31\n\t"
        "LSRS	r8, r4, #8\n\t"
        "LSRS	r9, r5, #8\n\t"
        "ORR	r9, r9, r4, LSL #24\n\t"
        "ORR	r8, r8, r5, LSL #24\n\t"
        "EOR	r7, r7, r9\n\t"
        "EOR	r6, r6, r8\n\t"
        "LSRS	r8, r4, #7\n\t"
        "LSRS	r9, r5, #7\n\t"
        "ORR	r8, r8, r5, LSL #25\n\t"
        "EOR	r7, r7, r9\n\t"
        "EOR	r6, r6, r8\n\t"
        "LDRD	r4, r5, [sp, #48]\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "STRD	r4, r5, [sp, #48]\n\t"
        /* Round 7 */
        "LDRD	r4, r5, [%[sha512], #40]\n\t"
        "LSRS	r6, r4, #14\n\t"
        "LSRS	r7, r5, #14\n\t"
        "ORR	r7, r7, r4, LSL #18\n\t"
        "ORR	r6, r6, r5, LSL #18\n\t"
        "LSRS	r8, r4, #18\n\t"
        "LSRS	r9, r5, #18\n\t"
        "ORR	r9, r9, r4, LSL #14\n\t"
        "ORR	r8, r8, r5, LSL #14\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "LSLS	r8, r4, #23\n\t"
        "LSLS	r9, r5, #23\n\t"
        "ORR	r9, r9, r4, LSR #9\n\t"
        "ORR	r8, r8, r5, LSR #9\n\t"
        "LDRD	r4, r5, [%[sha512]]\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "STRD	r4, r5, [%[sha512]]\n\t"
        "LDRD	r4, r5, [%[sha512], #40]\n\t"
        "LDRD	r6, r7, [%[sha512], #48]\n\t"
        "LDRD	r8, r9, [%[sha512], #56]\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "AND	r6, r6, r4\n\t"
        "AND	r7, r7, r5\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "LDRD	r4, r5, [%[sha512]]\n\t"
        "LDRD	r8, r9, [sp, #56]\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "LDRD	r6, r7, [r3, #56]\n\t"
        "ADDS	r4, r4, r8\n\t"
        "ADC	r5, r5, r9\n\t"
        "LDRD	r8, r9, [%[sha512], #32]\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "STRD	r4, r5, [%[sha512]]\n\t"
        "ADDS	r8, r8, r4\n\t"
        "ADC	r9, r9, r5\n\t"
        "LDRD	r4, r5, [%[sha512], #8]\n\t"
        "STRD	r8, r9, [%[sha512], #32]\n\t"
        "LSRS	r6, r4, #28\n\t"
        "LSRS	r7, r5, #28\n\t"
        "ORR	r7, r7, r4, LSL #4\n\t"
        "ORR	r6, r6, r5, LSL #4\n\t"
        "LSLS	r8, r4, #30\n\t"
        "LSLS	r9, r5, #30\n\t"
        "ORR	r9, r9, r4, LSR #2\n\t"
        "ORR	r8, r8, r5, LSR #2\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "LSLS	r8, r4, #25\n\t"
        "LSLS	r9, r5, #25\n\t"
        "ORR	r9, r9, r4, LSR #7\n\t"
        "ORR	r8, r8, r5, LSR #7\n\t"
        "LDRD	r4, r5, [%[sha512]]\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "LDRD	r8, r9, [%[sha512], #8]\n\t"
        "LDRD	r6, r7, [%[sha512], #16]\n\t"
        "STRD	r4, r5, [%[sha512]]\n\t"
        "EOR	r8, r8, r6\n\t"
        "EOR	r9, r9, r7\n\t"
        "AND	r10, r10, r8\n\t"
        "AND	r11, r11, r9\n\t"
        "EOR	r10, r10, r6\n\t"
        "EOR	r11, r11, r7\n\t"
        "LDRD	r6, r7, [%[sha512]]\n\t"
        "ADDS	r6, r6, r10\n\t"
        "ADC	r7, r7, r11\n\t"
        "STRD	r6, r7, [%[sha512]]\n\t"
        "MOV	r10, r8\n\t"
        "MOV	r11, r9\n\t"
        /* Calc new W[7] */
        "LDRD	r4, r5, [sp, #40]\n\t"
        "LSRS	r6, r4, #19\n\t"
        "LSRS	r7, r5, #19\n\t"
        "ORR	r7, r7, r4, LSL #13\n\t"
        "ORR	r6, r6, r5, LSL #13\n\t"
        "LSLS	r8, r4, #3\n\t"
        "LSLS	r9, r5, #3\n\t"
        "ORR	r9, r9, r4, LSR #29\n\t"
        "ORR	r8, r8, r5, LSR #29\n\t"
        "EOR	r7, r7, r9\n\t"
        "EOR	r6, r6, r8\n\t"
        "LSRS	r8, r4, #6\n\t"
        "LSRS	r9, r5, #6\n\t"
        "ORR	r8, r8, r5, LSL #26\n\t"
        "EOR	r7, r7, r9\n\t"
        "EOR	r6, r6, r8\n\t"
        "LDRD	r4, r5, [sp, #56]\n\t"
        "LDRD	r8, r9, [sp]\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "ADDS	r4, r4, r8\n\t"
        "ADC	r5, r5, r9\n\t"
        "STRD	r4, r5, [sp, #56]\n\t"
        "LDRD	r4, r5, [sp, #64]\n\t"
        "LSRS	r6, r4, #1\n\t"
        "LSRS	r7, r5, #1\n\t"
        "ORR	r7, r7, r4, LSL #31\n\t"
        "ORR	r6, r6, r5, LSL #31\n\t"
        "LSRS	r8, r4, #8\n\t"
        "LSRS	r9, r5, #8\n\t"
        "ORR	r9, r9, r4, LSL #24\n\t"
        "ORR	r8, r8, r5, LSL #24\n\t"
        "EOR	r7, r7, r9\n\t"
        "EOR	r6, r6, r8\n\t"
        "LSRS	r8, r4, #7\n\t"
        "LSRS	r9, r5, #7\n\t"
        "ORR	r8, r8, r5, LSL #25\n\t"
        "EOR	r7, r7, r9\n\t"
        "EOR	r6, r6, r8\n\t"
        "LDRD	r4, r5, [sp, #56]\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "STRD	r4, r5, [sp, #56]\n\t"
        /* Round 8 */
        "LDRD	r4, r5, [%[sha512], #32]\n\t"
        "LSRS	r6, r4, #14\n\t"
        "LSRS	r7, r5, #14\n\t"
        "ORR	r7, r7, r4, LSL #18\n\t"
        "ORR	r6, r6, r5, LSL #18\n\t"
        "LSRS	r8, r4, #18\n\t"
        "LSRS	r9, r5, #18\n\t"
        "ORR	r9, r9, r4, LSL #14\n\t"
        "ORR	r8, r8, r5, LSL #14\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "LSLS	r8, r4, #23\n\t"
        "LSLS	r9, r5, #23\n\t"
        "ORR	r9, r9, r4, LSR #9\n\t"
        "ORR	r8, r8, r5, LSR #9\n\t"
        "LDRD	r4, r5, [%[sha512], #56]\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "STRD	r4, r5, [%[sha512], #56]\n\t"
        "LDRD	r4, r5, [%[sha512], #32]\n\t"
        "LDRD	r6, r7, [%[sha512], #40]\n\t"
        "LDRD	r8, r9, [%[sha512], #48]\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "AND	r6, r6, r4\n\t"
        "AND	r7, r7, r5\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "LDRD	r4, r5, [%[sha512], #56]\n\t"
        "LDRD	r8, r9, [sp, #64]\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "LDRD	r6, r7, [r3, #64]\n\t"
        "ADDS	r4, r4, r8\n\t"
        "ADC	r5, r5, r9\n\t"
        "LDRD	r8, r9, [%[sha512], #24]\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "STRD	r4, r5, [%[sha512], #56]\n\t"
        "ADDS	r8, r8, r4\n\t"
        "ADC	r9, r9, r5\n\t"
        "LDRD	r4, r5, [%[sha512]]\n\t"
        "STRD	r8, r9, [%[sha512], #24]\n\t"
        "LSRS	r6, r4, #28\n\t"
        "LSRS	r7, r5, #28\n\t"
        "ORR	r7, r7, r4, LSL #4\n\t"
        "ORR	r6, r6, r5, LSL #4\n\t"
        "LSLS	r8, r4, #30\n\t"
        "LSLS	r9, r5, #30\n\t"
        "ORR	r9, r9, r4, LSR #2\n\t"
        "ORR	r8, r8, r5, LSR #2\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "LSLS	r8, r4, #25\n\t"
        "LSLS	r9, r5, #25\n\t"
        "ORR	r9, r9, r4, LSR #7\n\t"
        "ORR	r8, r8, r5, LSR #7\n\t"
        "LDRD	r4, r5, [%[sha512], #56]\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "LDRD	r8, r9, [%[sha512]]\n\t"
        "LDRD	r6, r7, [%[sha512], #8]\n\t"
        "STRD	r4, r5, [%[sha512], #56]\n\t"
        "EOR	r8, r8, r6\n\t"
        "EOR	r9, r9, r7\n\t"
        "AND	r10, r10, r8\n\t"
        "AND	r11, r11, r9\n\t"
        "EOR	r10, r10, r6\n\t"
        "EOR	r11, r11, r7\n\t"
        "LDRD	r6, r7, [%[sha512], #56]\n\t"
        "ADDS	r6, r6, r10\n\t"
        "ADC	r7, r7, r11\n\t"
        "STRD	r6, r7, [%[sha512], #56]\n\t"
        "MOV	r10, r8\n\t"
        "MOV	r11, r9\n\t"
        /* Calc new W[8] */
        "LDRD	r4, r5, [sp, #48]\n\t"
        "LSRS	r6, r4, #19\n\t"
        "LSRS	r7, r5, #19\n\t"
        "ORR	r7, r7, r4, LSL #13\n\t"
        "ORR	r6, r6, r5, LSL #13\n\t"
        "LSLS	r8, r4, #3\n\t"
        "LSLS	r9, r5, #3\n\t"
        "ORR	r9, r9, r4, LSR #29\n\t"
        "ORR	r8, r8, r5, LSR #29\n\t"
        "EOR	r7, r7, r9\n\t"
        "EOR	r6, r6, r8\n\t"
        "LSRS	r8, r4, #6\n\t"
        "LSRS	r9, r5, #6\n\t"
        "ORR	r8, r8, r5, LSL #26\n\t"
        "EOR	r7, r7, r9\n\t"
        "EOR	r6, r6, r8\n\t"
        "LDRD	r4, r5, [sp, #64]\n\t"
        "LDRD	r8, r9, [sp, #8]\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "ADDS	r4, r4, r8\n\t"
        "ADC	r5, r5, r9\n\t"
        "STRD	r4, r5, [sp, #64]\n\t"
        "LDRD	r4, r5, [sp, #72]\n\t"
        "LSRS	r6, r4, #1\n\t"
        "LSRS	r7, r5, #1\n\t"
        "ORR	r7, r7, r4, LSL #31\n\t"
        "ORR	r6, r6, r5, LSL #31\n\t"
        "LSRS	r8, r4, #8\n\t"
        "LSRS	r9, r5, #8\n\t"
        "ORR	r9, r9, r4, LSL #24\n\t"
        "ORR	r8, r8, r5, LSL #24\n\t"
        "EOR	r7, r7, r9\n\t"
        "EOR	r6, r6, r8\n\t"
        "LSRS	r8, r4, #7\n\t"
        "LSRS	r9, r5, #7\n\t"
        "ORR	r8, r8, r5, LSL #25\n\t"
        "EOR	r7, r7, r9\n\t"
        "EOR	r6, r6, r8\n\t"
        "LDRD	r4, r5, [sp, #64]\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "STRD	r4, r5, [sp, #64]\n\t"
        /* Round 9 */
        "LDRD	r4, r5, [%[sha512], #24]\n\t"
        "LSRS	r6, r4, #14\n\t"
        "LSRS	r7, r5, #14\n\t"
        "ORR	r7, r7, r4, LSL #18\n\t"
        "ORR	r6, r6, r5, LSL #18\n\t"
        "LSRS	r8, r4, #18\n\t"
        "LSRS	r9, r5, #18\n\t"
        "ORR	r9, r9, r4, LSL #14\n\t"
        "ORR	r8, r8, r5, LSL #14\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "LSLS	r8, r4, #23\n\t"
        "LSLS	r9, r5, #23\n\t"
        "ORR	r9, r9, r4, LSR #9\n\t"
        "ORR	r8, r8, r5, LSR #9\n\t"
        "LDRD	r4, r5, [%[sha512], #48]\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "STRD	r4, r5, [%[sha512], #48]\n\t"
        "LDRD	r4, r5, [%[sha512], #24]\n\t"
        "LDRD	r6, r7, [%[sha512], #32]\n\t"
        "LDRD	r8, r9, [%[sha512], #40]\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "AND	r6, r6, r4\n\t"
        "AND	r7, r7, r5\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "LDRD	r4, r5, [%[sha512], #48]\n\t"
        "LDRD	r8, r9, [sp, #72]\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "LDRD	r6, r7, [r3, #72]\n\t"
        "ADDS	r4, r4, r8\n\t"
        "ADC	r5, r5, r9\n\t"
        "LDRD	r8, r9, [%[sha512], #16]\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "STRD	r4, r5, [%[sha512], #48]\n\t"
        "ADDS	r8, r8, r4\n\t"
        "ADC	r9, r9, r5\n\t"
        "LDRD	r4, r5, [%[sha512], #56]\n\t"
        "STRD	r8, r9, [%[sha512], #16]\n\t"
        "LSRS	r6, r4, #28\n\t"
        "LSRS	r7, r5, #28\n\t"
        "ORR	r7, r7, r4, LSL #4\n\t"
        "ORR	r6, r6, r5, LSL #4\n\t"
        "LSLS	r8, r4, #30\n\t"
        "LSLS	r9, r5, #30\n\t"
        "ORR	r9, r9, r4, LSR #2\n\t"
        "ORR	r8, r8, r5, LSR #2\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "LSLS	r8, r4, #25\n\t"
        "LSLS	r9, r5, #25\n\t"
        "ORR	r9, r9, r4, LSR #7\n\t"
        "ORR	r8, r8, r5, LSR #7\n\t"
        "LDRD	r4, r5, [%[sha512], #48]\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "LDRD	r8, r9, [%[sha512], #56]\n\t"
        "LDRD	r6, r7, [%[sha512]]\n\t"
        "STRD	r4, r5, [%[sha512], #48]\n\t"
        "EOR	r8, r8, r6\n\t"
        "EOR	r9, r9, r7\n\t"
        "AND	r10, r10, r8\n\t"
        "AND	r11, r11, r9\n\t"
        "EOR	r10, r10, r6\n\t"
        "EOR	r11, r11, r7\n\t"
        "LDRD	r6, r7, [%[sha512], #48]\n\t"
        "ADDS	r6, r6, r10\n\t"
        "ADC	r7, r7, r11\n\t"
        "STRD	r6, r7, [%[sha512], #48]\n\t"
        "MOV	r10, r8\n\t"
        "MOV	r11, r9\n\t"
        /* Calc new W[9] */
        "LDRD	r4, r5, [sp, #56]\n\t"
        "LSRS	r6, r4, #19\n\t"
        "LSRS	r7, r5, #19\n\t"
        "ORR	r7, r7, r4, LSL #13\n\t"
        "ORR	r6, r6, r5, LSL #13\n\t"
        "LSLS	r8, r4, #3\n\t"
        "LSLS	r9, r5, #3\n\t"
        "ORR	r9, r9, r4, LSR #29\n\t"
        "ORR	r8, r8, r5, LSR #29\n\t"
        "EOR	r7, r7, r9\n\t"
        "EOR	r6, r6, r8\n\t"
        "LSRS	r8, r4, #6\n\t"
        "LSRS	r9, r5, #6\n\t"
        "ORR	r8, r8, r5, LSL #26\n\t"
        "EOR	r7, r7, r9\n\t"
        "EOR	r6, r6, r8\n\t"
        "LDRD	r4, r5, [sp, #72]\n\t"
        "LDRD	r8, r9, [sp, #16]\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "ADDS	r4, r4, r8\n\t"
        "ADC	r5, r5, r9\n\t"
        "STRD	r4, r5, [sp, #72]\n\t"
        "LDRD	r4, r5, [sp, #80]\n\t"
        "LSRS	r6, r4, #1\n\t"
        "LSRS	r7, r5, #1\n\t"
        "ORR	r7, r7, r4, LSL #31\n\t"
        "ORR	r6, r6, r5, LSL #31\n\t"
        "LSRS	r8, r4, #8\n\t"
        "LSRS	r9, r5, #8\n\t"
        "ORR	r9, r9, r4, LSL #24\n\t"
        "ORR	r8, r8, r5, LSL #24\n\t"
        "EOR	r7, r7, r9\n\t"
        "EOR	r6, r6, r8\n\t"
        "LSRS	r8, r4, #7\n\t"
        "LSRS	r9, r5, #7\n\t"
        "ORR	r8, r8, r5, LSL #25\n\t"
        "EOR	r7, r7, r9\n\t"
        "EOR	r6, r6, r8\n\t"
        "LDRD	r4, r5, [sp, #72]\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "STRD	r4, r5, [sp, #72]\n\t"
        /* Round 10 */
        "LDRD	r4, r5, [%[sha512], #16]\n\t"
        "LSRS	r6, r4, #14\n\t"
        "LSRS	r7, r5, #14\n\t"
        "ORR	r7, r7, r4, LSL #18\n\t"
        "ORR	r6, r6, r5, LSL #18\n\t"
        "LSRS	r8, r4, #18\n\t"
        "LSRS	r9, r5, #18\n\t"
        "ORR	r9, r9, r4, LSL #14\n\t"
        "ORR	r8, r8, r5, LSL #14\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "LSLS	r8, r4, #23\n\t"
        "LSLS	r9, r5, #23\n\t"
        "ORR	r9, r9, r4, LSR #9\n\t"
        "ORR	r8, r8, r5, LSR #9\n\t"
        "LDRD	r4, r5, [%[sha512], #40]\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "STRD	r4, r5, [%[sha512], #40]\n\t"
        "LDRD	r4, r5, [%[sha512], #16]\n\t"
        "LDRD	r6, r7, [%[sha512], #24]\n\t"
        "LDRD	r8, r9, [%[sha512], #32]\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "AND	r6, r6, r4\n\t"
        "AND	r7, r7, r5\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "LDRD	r4, r5, [%[sha512], #40]\n\t"
        "LDRD	r8, r9, [sp, #80]\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "LDRD	r6, r7, [r3, #80]\n\t"
        "ADDS	r4, r4, r8\n\t"
        "ADC	r5, r5, r9\n\t"
        "LDRD	r8, r9, [%[sha512], #8]\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "STRD	r4, r5, [%[sha512], #40]\n\t"
        "ADDS	r8, r8, r4\n\t"
        "ADC	r9, r9, r5\n\t"
        "LDRD	r4, r5, [%[sha512], #48]\n\t"
        "STRD	r8, r9, [%[sha512], #8]\n\t"
        "LSRS	r6, r4, #28\n\t"
        "LSRS	r7, r5, #28\n\t"
        "ORR	r7, r7, r4, LSL #4\n\t"
        "ORR	r6, r6, r5, LSL #4\n\t"
        "LSLS	r8, r4, #30\n\t"
        "LSLS	r9, r5, #30\n\t"
        "ORR	r9, r9, r4, LSR #2\n\t"
        "ORR	r8, r8, r5, LSR #2\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "LSLS	r8, r4, #25\n\t"
        "LSLS	r9, r5, #25\n\t"
        "ORR	r9, r9, r4, LSR #7\n\t"
        "ORR	r8, r8, r5, LSR #7\n\t"
        "LDRD	r4, r5, [%[sha512], #40]\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "LDRD	r8, r9, [%[sha512], #48]\n\t"
        "LDRD	r6, r7, [%[sha512], #56]\n\t"
        "STRD	r4, r5, [%[sha512], #40]\n\t"
        "EOR	r8, r8, r6\n\t"
        "EOR	r9, r9, r7\n\t"
        "AND	r10, r10, r8\n\t"
        "AND	r11, r11, r9\n\t"
        "EOR	r10, r10, r6\n\t"
        "EOR	r11, r11, r7\n\t"
        "LDRD	r6, r7, [%[sha512], #40]\n\t"
        "ADDS	r6, r6, r10\n\t"
        "ADC	r7, r7, r11\n\t"
        "STRD	r6, r7, [%[sha512], #40]\n\t"
        "MOV	r10, r8\n\t"
        "MOV	r11, r9\n\t"
        /* Calc new W[10] */
        "LDRD	r4, r5, [sp, #64]\n\t"
        "LSRS	r6, r4, #19\n\t"
        "LSRS	r7, r5, #19\n\t"
        "ORR	r7, r7, r4, LSL #13\n\t"
        "ORR	r6, r6, r5, LSL #13\n\t"
        "LSLS	r8, r4, #3\n\t"
        "LSLS	r9, r5, #3\n\t"
        "ORR	r9, r9, r4, LSR #29\n\t"
        "ORR	r8, r8, r5, LSR #29\n\t"
        "EOR	r7, r7, r9\n\t"
        "EOR	r6, r6, r8\n\t"
        "LSRS	r8, r4, #6\n\t"
        "LSRS	r9, r5, #6\n\t"
        "ORR	r8, r8, r5, LSL #26\n\t"
        "EOR	r7, r7, r9\n\t"
        "EOR	r6, r6, r8\n\t"
        "LDRD	r4, r5, [sp, #80]\n\t"
        "LDRD	r8, r9, [sp, #24]\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "ADDS	r4, r4, r8\n\t"
        "ADC	r5, r5, r9\n\t"
        "STRD	r4, r5, [sp, #80]\n\t"
        "LDRD	r4, r5, [sp, #88]\n\t"
        "LSRS	r6, r4, #1\n\t"
        "LSRS	r7, r5, #1\n\t"
        "ORR	r7, r7, r4, LSL #31\n\t"
        "ORR	r6, r6, r5, LSL #31\n\t"
        "LSRS	r8, r4, #8\n\t"
        "LSRS	r9, r5, #8\n\t"
        "ORR	r9, r9, r4, LSL #24\n\t"
        "ORR	r8, r8, r5, LSL #24\n\t"
        "EOR	r7, r7, r9\n\t"
        "EOR	r6, r6, r8\n\t"
        "LSRS	r8, r4, #7\n\t"
        "LSRS	r9, r5, #7\n\t"
        "ORR	r8, r8, r5, LSL #25\n\t"
        "EOR	r7, r7, r9\n\t"
        "EOR	r6, r6, r8\n\t"
        "LDRD	r4, r5, [sp, #80]\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "STRD	r4, r5, [sp, #80]\n\t"
        /* Round 11 */
        "LDRD	r4, r5, [%[sha512], #8]\n\t"
        "LSRS	r6, r4, #14\n\t"
        "LSRS	r7, r5, #14\n\t"
        "ORR	r7, r7, r4, LSL #18\n\t"
        "ORR	r6, r6, r5, LSL #18\n\t"
        "LSRS	r8, r4, #18\n\t"
        "LSRS	r9, r5, #18\n\t"
        "ORR	r9, r9, r4, LSL #14\n\t"
        "ORR	r8, r8, r5, LSL #14\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "LSLS	r8, r4, #23\n\t"
        "LSLS	r9, r5, #23\n\t"
        "ORR	r9, r9, r4, LSR #9\n\t"
        "ORR	r8, r8, r5, LSR #9\n\t"
        "LDRD	r4, r5, [%[sha512], #32]\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "STRD	r4, r5, [%[sha512], #32]\n\t"
        "LDRD	r4, r5, [%[sha512], #8]\n\t"
        "LDRD	r6, r7, [%[sha512], #16]\n\t"
        "LDRD	r8, r9, [%[sha512], #24]\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "AND	r6, r6, r4\n\t"
        "AND	r7, r7, r5\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "LDRD	r4, r5, [%[sha512], #32]\n\t"
        "LDRD	r8, r9, [sp, #88]\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "LDRD	r6, r7, [r3, #88]\n\t"
        "ADDS	r4, r4, r8\n\t"
        "ADC	r5, r5, r9\n\t"
        "LDRD	r8, r9, [%[sha512]]\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "STRD	r4, r5, [%[sha512], #32]\n\t"
        "ADDS	r8, r8, r4\n\t"
        "ADC	r9, r9, r5\n\t"
        "LDRD	r4, r5, [%[sha512], #40]\n\t"
        "STRD	r8, r9, [%[sha512]]\n\t"
        "LSRS	r6, r4, #28\n\t"
        "LSRS	r7, r5, #28\n\t"
        "ORR	r7, r7, r4, LSL #4\n\t"
        "ORR	r6, r6, r5, LSL #4\n\t"
        "LSLS	r8, r4, #30\n\t"
        "LSLS	r9, r5, #30\n\t"
        "ORR	r9, r9, r4, LSR #2\n\t"
        "ORR	r8, r8, r5, LSR #2\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "LSLS	r8, r4, #25\n\t"
        "LSLS	r9, r5, #25\n\t"
        "ORR	r9, r9, r4, LSR #7\n\t"
        "ORR	r8, r8, r5, LSR #7\n\t"
        "LDRD	r4, r5, [%[sha512], #32]\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "LDRD	r8, r9, [%[sha512], #40]\n\t"
        "LDRD	r6, r7, [%[sha512], #48]\n\t"
        "STRD	r4, r5, [%[sha512], #32]\n\t"
        "EOR	r8, r8, r6\n\t"
        "EOR	r9, r9, r7\n\t"
        "AND	r10, r10, r8\n\t"
        "AND	r11, r11, r9\n\t"
        "EOR	r10, r10, r6\n\t"
        "EOR	r11, r11, r7\n\t"
        "LDRD	r6, r7, [%[sha512], #32]\n\t"
        "ADDS	r6, r6, r10\n\t"
        "ADC	r7, r7, r11\n\t"
        "STRD	r6, r7, [%[sha512], #32]\n\t"
        "MOV	r10, r8\n\t"
        "MOV	r11, r9\n\t"
        /* Calc new W[11] */
        "LDRD	r4, r5, [sp, #72]\n\t"
        "LSRS	r6, r4, #19\n\t"
        "LSRS	r7, r5, #19\n\t"
        "ORR	r7, r7, r4, LSL #13\n\t"
        "ORR	r6, r6, r5, LSL #13\n\t"
        "LSLS	r8, r4, #3\n\t"
        "LSLS	r9, r5, #3\n\t"
        "ORR	r9, r9, r4, LSR #29\n\t"
        "ORR	r8, r8, r5, LSR #29\n\t"
        "EOR	r7, r7, r9\n\t"
        "EOR	r6, r6, r8\n\t"
        "LSRS	r8, r4, #6\n\t"
        "LSRS	r9, r5, #6\n\t"
        "ORR	r8, r8, r5, LSL #26\n\t"
        "EOR	r7, r7, r9\n\t"
        "EOR	r6, r6, r8\n\t"
        "LDRD	r4, r5, [sp, #88]\n\t"
        "LDRD	r8, r9, [sp, #32]\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "ADDS	r4, r4, r8\n\t"
        "ADC	r5, r5, r9\n\t"
        "STRD	r4, r5, [sp, #88]\n\t"
        "LDRD	r4, r5, [sp, #96]\n\t"
        "LSRS	r6, r4, #1\n\t"
        "LSRS	r7, r5, #1\n\t"
        "ORR	r7, r7, r4, LSL #31\n\t"
        "ORR	r6, r6, r5, LSL #31\n\t"
        "LSRS	r8, r4, #8\n\t"
        "LSRS	r9, r5, #8\n\t"
        "ORR	r9, r9, r4, LSL #24\n\t"
        "ORR	r8, r8, r5, LSL #24\n\t"
        "EOR	r7, r7, r9\n\t"
        "EOR	r6, r6, r8\n\t"
        "LSRS	r8, r4, #7\n\t"
        "LSRS	r9, r5, #7\n\t"
        "ORR	r8, r8, r5, LSL #25\n\t"
        "EOR	r7, r7, r9\n\t"
        "EOR	r6, r6, r8\n\t"
        "LDRD	r4, r5, [sp, #88]\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "STRD	r4, r5, [sp, #88]\n\t"
        /* Round 12 */
        "LDRD	r4, r5, [%[sha512]]\n\t"
        "LSRS	r6, r4, #14\n\t"
        "LSRS	r7, r5, #14\n\t"
        "ORR	r7, r7, r4, LSL #18\n\t"
        "ORR	r6, r6, r5, LSL #18\n\t"
        "LSRS	r8, r4, #18\n\t"
        "LSRS	r9, r5, #18\n\t"
        "ORR	r9, r9, r4, LSL #14\n\t"
        "ORR	r8, r8, r5, LSL #14\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "LSLS	r8, r4, #23\n\t"
        "LSLS	r9, r5, #23\n\t"
        "ORR	r9, r9, r4, LSR #9\n\t"
        "ORR	r8, r8, r5, LSR #9\n\t"
        "LDRD	r4, r5, [%[sha512], #24]\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "STRD	r4, r5, [%[sha512], #24]\n\t"
        "LDRD	r4, r5, [%[sha512]]\n\t"
        "LDRD	r6, r7, [%[sha512], #8]\n\t"
        "LDRD	r8, r9, [%[sha512], #16]\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "AND	r6, r6, r4\n\t"
        "AND	r7, r7, r5\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "LDRD	r4, r5, [%[sha512], #24]\n\t"
        "LDRD	r8, r9, [sp, #96]\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "LDRD	r6, r7, [r3, #96]\n\t"
        "ADDS	r4, r4, r8\n\t"
        "ADC	r5, r5, r9\n\t"
        "LDRD	r8, r9, [%[sha512], #56]\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "STRD	r4, r5, [%[sha512], #24]\n\t"
        "ADDS	r8, r8, r4\n\t"
        "ADC	r9, r9, r5\n\t"
        "LDRD	r4, r5, [%[sha512], #32]\n\t"
        "STRD	r8, r9, [%[sha512], #56]\n\t"
        "LSRS	r6, r4, #28\n\t"
        "LSRS	r7, r5, #28\n\t"
        "ORR	r7, r7, r4, LSL #4\n\t"
        "ORR	r6, r6, r5, LSL #4\n\t"
        "LSLS	r8, r4, #30\n\t"
        "LSLS	r9, r5, #30\n\t"
        "ORR	r9, r9, r4, LSR #2\n\t"
        "ORR	r8, r8, r5, LSR #2\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "LSLS	r8, r4, #25\n\t"
        "LSLS	r9, r5, #25\n\t"
        "ORR	r9, r9, r4, LSR #7\n\t"
        "ORR	r8, r8, r5, LSR #7\n\t"
        "LDRD	r4, r5, [%[sha512], #24]\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "LDRD	r8, r9, [%[sha512], #32]\n\t"
        "LDRD	r6, r7, [%[sha512], #40]\n\t"
        "STRD	r4, r5, [%[sha512], #24]\n\t"
        "EOR	r8, r8, r6\n\t"
        "EOR	r9, r9, r7\n\t"
        "AND	r10, r10, r8\n\t"
        "AND	r11, r11, r9\n\t"
        "EOR	r10, r10, r6\n\t"
        "EOR	r11, r11, r7\n\t"
        "LDRD	r6, r7, [%[sha512], #24]\n\t"
        "ADDS	r6, r6, r10\n\t"
        "ADC	r7, r7, r11\n\t"
        "STRD	r6, r7, [%[sha512], #24]\n\t"
        "MOV	r10, r8\n\t"
        "MOV	r11, r9\n\t"
        /* Calc new W[12] */
        "LDRD	r4, r5, [sp, #80]\n\t"
        "LSRS	r6, r4, #19\n\t"
        "LSRS	r7, r5, #19\n\t"
        "ORR	r7, r7, r4, LSL #13\n\t"
        "ORR	r6, r6, r5, LSL #13\n\t"
        "LSLS	r8, r4, #3\n\t"
        "LSLS	r9, r5, #3\n\t"
        "ORR	r9, r9, r4, LSR #29\n\t"
        "ORR	r8, r8, r5, LSR #29\n\t"
        "EOR	r7, r7, r9\n\t"
        "EOR	r6, r6, r8\n\t"
        "LSRS	r8, r4, #6\n\t"
        "LSRS	r9, r5, #6\n\t"
        "ORR	r8, r8, r5, LSL #26\n\t"
        "EOR	r7, r7, r9\n\t"
        "EOR	r6, r6, r8\n\t"
        "LDRD	r4, r5, [sp, #96]\n\t"
        "LDRD	r8, r9, [sp, #40]\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "ADDS	r4, r4, r8\n\t"
        "ADC	r5, r5, r9\n\t"
        "STRD	r4, r5, [sp, #96]\n\t"
        "LDRD	r4, r5, [sp, #104]\n\t"
        "LSRS	r6, r4, #1\n\t"
        "LSRS	r7, r5, #1\n\t"
        "ORR	r7, r7, r4, LSL #31\n\t"
        "ORR	r6, r6, r5, LSL #31\n\t"
        "LSRS	r8, r4, #8\n\t"
        "LSRS	r9, r5, #8\n\t"
        "ORR	r9, r9, r4, LSL #24\n\t"
        "ORR	r8, r8, r5, LSL #24\n\t"
        "EOR	r7, r7, r9\n\t"
        "EOR	r6, r6, r8\n\t"
        "LSRS	r8, r4, #7\n\t"
        "LSRS	r9, r5, #7\n\t"
        "ORR	r8, r8, r5, LSL #25\n\t"
        "EOR	r7, r7, r9\n\t"
        "EOR	r6, r6, r8\n\t"
        "LDRD	r4, r5, [sp, #96]\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "STRD	r4, r5, [sp, #96]\n\t"
        /* Round 13 */
        "LDRD	r4, r5, [%[sha512], #56]\n\t"
        "LSRS	r6, r4, #14\n\t"
        "LSRS	r7, r5, #14\n\t"
        "ORR	r7, r7, r4, LSL #18\n\t"
        "ORR	r6, r6, r5, LSL #18\n\t"
        "LSRS	r8, r4, #18\n\t"
        "LSRS	r9, r5, #18\n\t"
        "ORR	r9, r9, r4, LSL #14\n\t"
        "ORR	r8, r8, r5, LSL #14\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "LSLS	r8, r4, #23\n\t"
        "LSLS	r9, r5, #23\n\t"
        "ORR	r9, r9, r4, LSR #9\n\t"
        "ORR	r8, r8, r5, LSR #9\n\t"
        "LDRD	r4, r5, [%[sha512], #16]\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "STRD	r4, r5, [%[sha512], #16]\n\t"
        "LDRD	r4, r5, [%[sha512], #56]\n\t"
        "LDRD	r6, r7, [%[sha512]]\n\t"
        "LDRD	r8, r9, [%[sha512], #8]\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "AND	r6, r6, r4\n\t"
        "AND	r7, r7, r5\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "LDRD	r4, r5, [%[sha512], #16]\n\t"
        "LDRD	r8, r9, [sp, #104]\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "LDRD	r6, r7, [r3, #104]\n\t"
        "ADDS	r4, r4, r8\n\t"
        "ADC	r5, r5, r9\n\t"
        "LDRD	r8, r9, [%[sha512], #48]\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "STRD	r4, r5, [%[sha512], #16]\n\t"
        "ADDS	r8, r8, r4\n\t"
        "ADC	r9, r9, r5\n\t"
        "LDRD	r4, r5, [%[sha512], #24]\n\t"
        "STRD	r8, r9, [%[sha512], #48]\n\t"
        "LSRS	r6, r4, #28\n\t"
        "LSRS	r7, r5, #28\n\t"
        "ORR	r7, r7, r4, LSL #4\n\t"
        "ORR	r6, r6, r5, LSL #4\n\t"
        "LSLS	r8, r4, #30\n\t"
        "LSLS	r9, r5, #30\n\t"
        "ORR	r9, r9, r4, LSR #2\n\t"
        "ORR	r8, r8, r5, LSR #2\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "LSLS	r8, r4, #25\n\t"
        "LSLS	r9, r5, #25\n\t"
        "ORR	r9, r9, r4, LSR #7\n\t"
        "ORR	r8, r8, r5, LSR #7\n\t"
        "LDRD	r4, r5, [%[sha512], #16]\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "LDRD	r8, r9, [%[sha512], #24]\n\t"
        "LDRD	r6, r7, [%[sha512], #32]\n\t"
        "STRD	r4, r5, [%[sha512], #16]\n\t"
        "EOR	r8, r8, r6\n\t"
        "EOR	r9, r9, r7\n\t"
        "AND	r10, r10, r8\n\t"
        "AND	r11, r11, r9\n\t"
        "EOR	r10, r10, r6\n\t"
        "EOR	r11, r11, r7\n\t"
        "LDRD	r6, r7, [%[sha512], #16]\n\t"
        "ADDS	r6, r6, r10\n\t"
        "ADC	r7, r7, r11\n\t"
        "STRD	r6, r7, [%[sha512], #16]\n\t"
        "MOV	r10, r8\n\t"
        "MOV	r11, r9\n\t"
        /* Calc new W[13] */
        "LDRD	r4, r5, [sp, #88]\n\t"
        "LSRS	r6, r4, #19\n\t"
        "LSRS	r7, r5, #19\n\t"
        "ORR	r7, r7, r4, LSL #13\n\t"
        "ORR	r6, r6, r5, LSL #13\n\t"
        "LSLS	r8, r4, #3\n\t"
        "LSLS	r9, r5, #3\n\t"
        "ORR	r9, r9, r4, LSR #29\n\t"
        "ORR	r8, r8, r5, LSR #29\n\t"
        "EOR	r7, r7, r9\n\t"
        "EOR	r6, r6, r8\n\t"
        "LSRS	r8, r4, #6\n\t"
        "LSRS	r9, r5, #6\n\t"
        "ORR	r8, r8, r5, LSL #26\n\t"
        "EOR	r7, r7, r9\n\t"
        "EOR	r6, r6, r8\n\t"
        "LDRD	r4, r5, [sp, #104]\n\t"
        "LDRD	r8, r9, [sp, #48]\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "ADDS	r4, r4, r8\n\t"
        "ADC	r5, r5, r9\n\t"
        "STRD	r4, r5, [sp, #104]\n\t"
        "LDRD	r4, r5, [sp, #112]\n\t"
        "LSRS	r6, r4, #1\n\t"
        "LSRS	r7, r5, #1\n\t"
        "ORR	r7, r7, r4, LSL #31\n\t"
        "ORR	r6, r6, r5, LSL #31\n\t"
        "LSRS	r8, r4, #8\n\t"
        "LSRS	r9, r5, #8\n\t"
        "ORR	r9, r9, r4, LSL #24\n\t"
        "ORR	r8, r8, r5, LSL #24\n\t"
        "EOR	r7, r7, r9\n\t"
        "EOR	r6, r6, r8\n\t"
        "LSRS	r8, r4, #7\n\t"
        "LSRS	r9, r5, #7\n\t"
        "ORR	r8, r8, r5, LSL #25\n\t"
        "EOR	r7, r7, r9\n\t"
        "EOR	r6, r6, r8\n\t"
        "LDRD	r4, r5, [sp, #104]\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "STRD	r4, r5, [sp, #104]\n\t"
        /* Round 14 */
        "LDRD	r4, r5, [%[sha512], #48]\n\t"
        "LSRS	r6, r4, #14\n\t"
        "LSRS	r7, r5, #14\n\t"
        "ORR	r7, r7, r4, LSL #18\n\t"
        "ORR	r6, r6, r5, LSL #18\n\t"
        "LSRS	r8, r4, #18\n\t"
        "LSRS	r9, r5, #18\n\t"
        "ORR	r9, r9, r4, LSL #14\n\t"
        "ORR	r8, r8, r5, LSL #14\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "LSLS	r8, r4, #23\n\t"
        "LSLS	r9, r5, #23\n\t"
        "ORR	r9, r9, r4, LSR #9\n\t"
        "ORR	r8, r8, r5, LSR #9\n\t"
        "LDRD	r4, r5, [%[sha512], #8]\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "STRD	r4, r5, [%[sha512], #8]\n\t"
        "LDRD	r4, r5, [%[sha512], #48]\n\t"
        "LDRD	r6, r7, [%[sha512], #56]\n\t"
        "LDRD	r8, r9, [%[sha512]]\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "AND	r6, r6, r4\n\t"
        "AND	r7, r7, r5\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "LDRD	r4, r5, [%[sha512], #8]\n\t"
        "LDRD	r8, r9, [sp, #112]\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "LDRD	r6, r7, [r3, #112]\n\t"
        "ADDS	r4, r4, r8\n\t"
        "ADC	r5, r5, r9\n\t"
        "LDRD	r8, r9, [%[sha512], #40]\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "STRD	r4, r5, [%[sha512], #8]\n\t"
        "ADDS	r8, r8, r4\n\t"
        "ADC	r9, r9, r5\n\t"
        "LDRD	r4, r5, [%[sha512], #16]\n\t"
        "STRD	r8, r9, [%[sha512], #40]\n\t"
        "LSRS	r6, r4, #28\n\t"
        "LSRS	r7, r5, #28\n\t"
        "ORR	r7, r7, r4, LSL #4\n\t"
        "ORR	r6, r6, r5, LSL #4\n\t"
        "LSLS	r8, r4, #30\n\t"
        "LSLS	r9, r5, #30\n\t"
        "ORR	r9, r9, r4, LSR #2\n\t"
        "ORR	r8, r8, r5, LSR #2\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "LSLS	r8, r4, #25\n\t"
        "LSLS	r9, r5, #25\n\t"
        "ORR	r9, r9, r4, LSR #7\n\t"
        "ORR	r8, r8, r5, LSR #7\n\t"
        "LDRD	r4, r5, [%[sha512], #8]\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "LDRD	r8, r9, [%[sha512], #16]\n\t"
        "LDRD	r6, r7, [%[sha512], #24]\n\t"
        "STRD	r4, r5, [%[sha512], #8]\n\t"
        "EOR	r8, r8, r6\n\t"
        "EOR	r9, r9, r7\n\t"
        "AND	r10, r10, r8\n\t"
        "AND	r11, r11, r9\n\t"
        "EOR	r10, r10, r6\n\t"
        "EOR	r11, r11, r7\n\t"
        "LDRD	r6, r7, [%[sha512], #8]\n\t"
        "ADDS	r6, r6, r10\n\t"
        "ADC	r7, r7, r11\n\t"
        "STRD	r6, r7, [%[sha512], #8]\n\t"
        "MOV	r10, r8\n\t"
        "MOV	r11, r9\n\t"
        /* Calc new W[14] */
        "LDRD	r4, r5, [sp, #96]\n\t"
        "LSRS	r6, r4, #19\n\t"
        "LSRS	r7, r5, #19\n\t"
        "ORR	r7, r7, r4, LSL #13\n\t"
        "ORR	r6, r6, r5, LSL #13\n\t"
        "LSLS	r8, r4, #3\n\t"
        "LSLS	r9, r5, #3\n\t"
        "ORR	r9, r9, r4, LSR #29\n\t"
        "ORR	r8, r8, r5, LSR #29\n\t"
        "EOR	r7, r7, r9\n\t"
        "EOR	r6, r6, r8\n\t"
        "LSRS	r8, r4, #6\n\t"
        "LSRS	r9, r5, #6\n\t"
        "ORR	r8, r8, r5, LSL #26\n\t"
        "EOR	r7, r7, r9\n\t"
        "EOR	r6, r6, r8\n\t"
        "LDRD	r4, r5, [sp, #112]\n\t"
        "LDRD	r8, r9, [sp, #56]\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "ADDS	r4, r4, r8\n\t"
        "ADC	r5, r5, r9\n\t"
        "STRD	r4, r5, [sp, #112]\n\t"
        "LDRD	r4, r5, [sp, #120]\n\t"
        "LSRS	r6, r4, #1\n\t"
        "LSRS	r7, r5, #1\n\t"
        "ORR	r7, r7, r4, LSL #31\n\t"
        "ORR	r6, r6, r5, LSL #31\n\t"
        "LSRS	r8, r4, #8\n\t"
        "LSRS	r9, r5, #8\n\t"
        "ORR	r9, r9, r4, LSL #24\n\t"
        "ORR	r8, r8, r5, LSL #24\n\t"
        "EOR	r7, r7, r9\n\t"
        "EOR	r6, r6, r8\n\t"
        "LSRS	r8, r4, #7\n\t"
        "LSRS	r9, r5, #7\n\t"
        "ORR	r8, r8, r5, LSL #25\n\t"
        "EOR	r7, r7, r9\n\t"
        "EOR	r6, r6, r8\n\t"
        "LDRD	r4, r5, [sp, #112]\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "STRD	r4, r5, [sp, #112]\n\t"
        /* Round 15 */
        "LDRD	r4, r5, [%[sha512], #40]\n\t"
        "LSRS	r6, r4, #14\n\t"
        "LSRS	r7, r5, #14\n\t"
        "ORR	r7, r7, r4, LSL #18\n\t"
        "ORR	r6, r6, r5, LSL #18\n\t"
        "LSRS	r8, r4, #18\n\t"
        "LSRS	r9, r5, #18\n\t"
        "ORR	r9, r9, r4, LSL #14\n\t"
        "ORR	r8, r8, r5, LSL #14\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "LSLS	r8, r4, #23\n\t"
        "LSLS	r9, r5, #23\n\t"
        "ORR	r9, r9, r4, LSR #9\n\t"
        "ORR	r8, r8, r5, LSR #9\n\t"
        "LDRD	r4, r5, [%[sha512]]\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "STRD	r4, r5, [%[sha512]]\n\t"
        "LDRD	r4, r5, [%[sha512], #40]\n\t"
        "LDRD	r6, r7, [%[sha512], #48]\n\t"
        "LDRD	r8, r9, [%[sha512], #56]\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "AND	r6, r6, r4\n\t"
        "AND	r7, r7, r5\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "LDRD	r4, r5, [%[sha512]]\n\t"
        "LDRD	r8, r9, [sp, #120]\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "LDRD	r6, r7, [r3, #120]\n\t"
        "ADDS	r4, r4, r8\n\t"
        "ADC	r5, r5, r9\n\t"
        "LDRD	r8, r9, [%[sha512], #32]\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "STRD	r4, r5, [%[sha512]]\n\t"
        "ADDS	r8, r8, r4\n\t"
        "ADC	r9, r9, r5\n\t"
        "LDRD	r4, r5, [%[sha512], #8]\n\t"
        "STRD	r8, r9, [%[sha512], #32]\n\t"
        "LSRS	r6, r4, #28\n\t"
        "LSRS	r7, r5, #28\n\t"
        "ORR	r7, r7, r4, LSL #4\n\t"
        "ORR	r6, r6, r5, LSL #4\n\t"
        "LSLS	r8, r4, #30\n\t"
        "LSLS	r9, r5, #30\n\t"
        "ORR	r9, r9, r4, LSR #2\n\t"
        "ORR	r8, r8, r5, LSR #2\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "LSLS	r8, r4, #25\n\t"
        "LSLS	r9, r5, #25\n\t"
        "ORR	r9, r9, r4, LSR #7\n\t"
        "ORR	r8, r8, r5, LSR #7\n\t"
        "LDRD	r4, r5, [%[sha512]]\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "LDRD	r8, r9, [%[sha512], #8]\n\t"
        "LDRD	r6, r7, [%[sha512], #16]\n\t"
        "STRD	r4, r5, [%[sha512]]\n\t"
        "EOR	r8, r8, r6\n\t"
        "EOR	r9, r9, r7\n\t"
        "AND	r10, r10, r8\n\t"
        "AND	r11, r11, r9\n\t"
        "EOR	r10, r10, r6\n\t"
        "EOR	r11, r11, r7\n\t"
        "LDRD	r6, r7, [%[sha512]]\n\t"
        "ADDS	r6, r6, r10\n\t"
        "ADC	r7, r7, r11\n\t"
        "STRD	r6, r7, [%[sha512]]\n\t"
        "MOV	r10, r8\n\t"
        "MOV	r11, r9\n\t"
        /* Calc new W[15] */
        "LDRD	r4, r5, [sp, #104]\n\t"
        "LSRS	r6, r4, #19\n\t"
        "LSRS	r7, r5, #19\n\t"
        "ORR	r7, r7, r4, LSL #13\n\t"
        "ORR	r6, r6, r5, LSL #13\n\t"
        "LSLS	r8, r4, #3\n\t"
        "LSLS	r9, r5, #3\n\t"
        "ORR	r9, r9, r4, LSR #29\n\t"
        "ORR	r8, r8, r5, LSR #29\n\t"
        "EOR	r7, r7, r9\n\t"
        "EOR	r6, r6, r8\n\t"
        "LSRS	r8, r4, #6\n\t"
        "LSRS	r9, r5, #6\n\t"
        "ORR	r8, r8, r5, LSL #26\n\t"
        "EOR	r7, r7, r9\n\t"
        "EOR	r6, r6, r8\n\t"
        "LDRD	r4, r5, [sp, #120]\n\t"
        "LDRD	r8, r9, [sp, #64]\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "ADDS	r4, r4, r8\n\t"
        "ADC	r5, r5, r9\n\t"
        "STRD	r4, r5, [sp, #120]\n\t"
        "LDRD	r4, r5, [sp]\n\t"
        "LSRS	r6, r4, #1\n\t"
        "LSRS	r7, r5, #1\n\t"
        "ORR	r7, r7, r4, LSL #31\n\t"
        "ORR	r6, r6, r5, LSL #31\n\t"
        "LSRS	r8, r4, #8\n\t"
        "LSRS	r9, r5, #8\n\t"
        "ORR	r9, r9, r4, LSL #24\n\t"
        "ORR	r8, r8, r5, LSL #24\n\t"
        "EOR	r7, r7, r9\n\t"
        "EOR	r6, r6, r8\n\t"
        "LSRS	r8, r4, #7\n\t"
        "LSRS	r9, r5, #7\n\t"
        "ORR	r8, r8, r5, LSL #25\n\t"
        "EOR	r7, r7, r9\n\t"
        "EOR	r6, r6, r8\n\t"
        "LDRD	r4, r5, [sp, #120]\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "STRD	r4, r5, [sp, #120]\n\t"
        "ADD	r3, r3, #0x80\n\t"
        "SUBS	r12, r12, #0x1\n\t"
#ifdef __GNUC__
        "BNE	L_SHA512_transform_len_start%=\n\t"
#else
        "BNE.W	L_SHA512_transform_len_start%=\n\t"
#endif
        /* Round 0 */
        "LDRD	r4, r5, [%[sha512], #32]\n\t"
        "LSRS	r6, r4, #14\n\t"
        "LSRS	r7, r5, #14\n\t"
        "ORR	r7, r7, r4, LSL #18\n\t"
        "ORR	r6, r6, r5, LSL #18\n\t"
        "LSRS	r8, r4, #18\n\t"
        "LSRS	r9, r5, #18\n\t"
        "ORR	r9, r9, r4, LSL #14\n\t"
        "ORR	r8, r8, r5, LSL #14\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "LSLS	r8, r4, #23\n\t"
        "LSLS	r9, r5, #23\n\t"
        "ORR	r9, r9, r4, LSR #9\n\t"
        "ORR	r8, r8, r5, LSR #9\n\t"
        "LDRD	r4, r5, [%[sha512], #56]\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "STRD	r4, r5, [%[sha512], #56]\n\t"
        "LDRD	r4, r5, [%[sha512], #32]\n\t"
        "LDRD	r6, r7, [%[sha512], #40]\n\t"
        "LDRD	r8, r9, [%[sha512], #48]\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "AND	r6, r6, r4\n\t"
        "AND	r7, r7, r5\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "LDRD	r4, r5, [%[sha512], #56]\n\t"
        "LDRD	r8, r9, [sp]\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "LDRD	r6, r7, [r3]\n\t"
        "ADDS	r4, r4, r8\n\t"
        "ADC	r5, r5, r9\n\t"
        "LDRD	r8, r9, [%[sha512], #24]\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "STRD	r4, r5, [%[sha512], #56]\n\t"
        "ADDS	r8, r8, r4\n\t"
        "ADC	r9, r9, r5\n\t"
        "LDRD	r4, r5, [%[sha512]]\n\t"
        "STRD	r8, r9, [%[sha512], #24]\n\t"
        "LSRS	r6, r4, #28\n\t"
        "LSRS	r7, r5, #28\n\t"
        "ORR	r7, r7, r4, LSL #4\n\t"
        "ORR	r6, r6, r5, LSL #4\n\t"
        "LSLS	r8, r4, #30\n\t"
        "LSLS	r9, r5, #30\n\t"
        "ORR	r9, r9, r4, LSR #2\n\t"
        "ORR	r8, r8, r5, LSR #2\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "LSLS	r8, r4, #25\n\t"
        "LSLS	r9, r5, #25\n\t"
        "ORR	r9, r9, r4, LSR #7\n\t"
        "ORR	r8, r8, r5, LSR #7\n\t"
        "LDRD	r4, r5, [%[sha512], #56]\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "LDRD	r8, r9, [%[sha512]]\n\t"
        "LDRD	r6, r7, [%[sha512], #8]\n\t"
        "STRD	r4, r5, [%[sha512], #56]\n\t"
        "EOR	r8, r8, r6\n\t"
        "EOR	r9, r9, r7\n\t"
        "AND	r10, r10, r8\n\t"
        "AND	r11, r11, r9\n\t"
        "EOR	r10, r10, r6\n\t"
        "EOR	r11, r11, r7\n\t"
        "LDRD	r6, r7, [%[sha512], #56]\n\t"
        "ADDS	r6, r6, r10\n\t"
        "ADC	r7, r7, r11\n\t"
        "STRD	r6, r7, [%[sha512], #56]\n\t"
        "MOV	r10, r8\n\t"
        "MOV	r11, r9\n\t"
        /* Round 1 */
        "LDRD	r4, r5, [%[sha512], #24]\n\t"
        "LSRS	r6, r4, #14\n\t"
        "LSRS	r7, r5, #14\n\t"
        "ORR	r7, r7, r4, LSL #18\n\t"
        "ORR	r6, r6, r5, LSL #18\n\t"
        "LSRS	r8, r4, #18\n\t"
        "LSRS	r9, r5, #18\n\t"
        "ORR	r9, r9, r4, LSL #14\n\t"
        "ORR	r8, r8, r5, LSL #14\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "LSLS	r8, r4, #23\n\t"
        "LSLS	r9, r5, #23\n\t"
        "ORR	r9, r9, r4, LSR #9\n\t"
        "ORR	r8, r8, r5, LSR #9\n\t"
        "LDRD	r4, r5, [%[sha512], #48]\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "STRD	r4, r5, [%[sha512], #48]\n\t"
        "LDRD	r4, r5, [%[sha512], #24]\n\t"
        "LDRD	r6, r7, [%[sha512], #32]\n\t"
        "LDRD	r8, r9, [%[sha512], #40]\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "AND	r6, r6, r4\n\t"
        "AND	r7, r7, r5\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "LDRD	r4, r5, [%[sha512], #48]\n\t"
        "LDRD	r8, r9, [sp, #8]\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "LDRD	r6, r7, [r3, #8]\n\t"
        "ADDS	r4, r4, r8\n\t"
        "ADC	r5, r5, r9\n\t"
        "LDRD	r8, r9, [%[sha512], #16]\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "STRD	r4, r5, [%[sha512], #48]\n\t"
        "ADDS	r8, r8, r4\n\t"
        "ADC	r9, r9, r5\n\t"
        "LDRD	r4, r5, [%[sha512], #56]\n\t"
        "STRD	r8, r9, [%[sha512], #16]\n\t"
        "LSRS	r6, r4, #28\n\t"
        "LSRS	r7, r5, #28\n\t"
        "ORR	r7, r7, r4, LSL #4\n\t"
        "ORR	r6, r6, r5, LSL #4\n\t"
        "LSLS	r8, r4, #30\n\t"
        "LSLS	r9, r5, #30\n\t"
        "ORR	r9, r9, r4, LSR #2\n\t"
        "ORR	r8, r8, r5, LSR #2\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "LSLS	r8, r4, #25\n\t"
        "LSLS	r9, r5, #25\n\t"
        "ORR	r9, r9, r4, LSR #7\n\t"
        "ORR	r8, r8, r5, LSR #7\n\t"
        "LDRD	r4, r5, [%[sha512], #48]\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "LDRD	r8, r9, [%[sha512], #56]\n\t"
        "LDRD	r6, r7, [%[sha512]]\n\t"
        "STRD	r4, r5, [%[sha512], #48]\n\t"
        "EOR	r8, r8, r6\n\t"
        "EOR	r9, r9, r7\n\t"
        "AND	r10, r10, r8\n\t"
        "AND	r11, r11, r9\n\t"
        "EOR	r10, r10, r6\n\t"
        "EOR	r11, r11, r7\n\t"
        "LDRD	r6, r7, [%[sha512], #48]\n\t"
        "ADDS	r6, r6, r10\n\t"
        "ADC	r7, r7, r11\n\t"
        "STRD	r6, r7, [%[sha512], #48]\n\t"
        "MOV	r10, r8\n\t"
        "MOV	r11, r9\n\t"
        /* Round 2 */
        "LDRD	r4, r5, [%[sha512], #16]\n\t"
        "LSRS	r6, r4, #14\n\t"
        "LSRS	r7, r5, #14\n\t"
        "ORR	r7, r7, r4, LSL #18\n\t"
        "ORR	r6, r6, r5, LSL #18\n\t"
        "LSRS	r8, r4, #18\n\t"
        "LSRS	r9, r5, #18\n\t"
        "ORR	r9, r9, r4, LSL #14\n\t"
        "ORR	r8, r8, r5, LSL #14\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "LSLS	r8, r4, #23\n\t"
        "LSLS	r9, r5, #23\n\t"
        "ORR	r9, r9, r4, LSR #9\n\t"
        "ORR	r8, r8, r5, LSR #9\n\t"
        "LDRD	r4, r5, [%[sha512], #40]\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "STRD	r4, r5, [%[sha512], #40]\n\t"
        "LDRD	r4, r5, [%[sha512], #16]\n\t"
        "LDRD	r6, r7, [%[sha512], #24]\n\t"
        "LDRD	r8, r9, [%[sha512], #32]\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "AND	r6, r6, r4\n\t"
        "AND	r7, r7, r5\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "LDRD	r4, r5, [%[sha512], #40]\n\t"
        "LDRD	r8, r9, [sp, #16]\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "LDRD	r6, r7, [r3, #16]\n\t"
        "ADDS	r4, r4, r8\n\t"
        "ADC	r5, r5, r9\n\t"
        "LDRD	r8, r9, [%[sha512], #8]\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "STRD	r4, r5, [%[sha512], #40]\n\t"
        "ADDS	r8, r8, r4\n\t"
        "ADC	r9, r9, r5\n\t"
        "LDRD	r4, r5, [%[sha512], #48]\n\t"
        "STRD	r8, r9, [%[sha512], #8]\n\t"
        "LSRS	r6, r4, #28\n\t"
        "LSRS	r7, r5, #28\n\t"
        "ORR	r7, r7, r4, LSL #4\n\t"
        "ORR	r6, r6, r5, LSL #4\n\t"
        "LSLS	r8, r4, #30\n\t"
        "LSLS	r9, r5, #30\n\t"
        "ORR	r9, r9, r4, LSR #2\n\t"
        "ORR	r8, r8, r5, LSR #2\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "LSLS	r8, r4, #25\n\t"
        "LSLS	r9, r5, #25\n\t"
        "ORR	r9, r9, r4, LSR #7\n\t"
        "ORR	r8, r8, r5, LSR #7\n\t"
        "LDRD	r4, r5, [%[sha512], #40]\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "LDRD	r8, r9, [%[sha512], #48]\n\t"
        "LDRD	r6, r7, [%[sha512], #56]\n\t"
        "STRD	r4, r5, [%[sha512], #40]\n\t"
        "EOR	r8, r8, r6\n\t"
        "EOR	r9, r9, r7\n\t"
        "AND	r10, r10, r8\n\t"
        "AND	r11, r11, r9\n\t"
        "EOR	r10, r10, r6\n\t"
        "EOR	r11, r11, r7\n\t"
        "LDRD	r6, r7, [%[sha512], #40]\n\t"
        "ADDS	r6, r6, r10\n\t"
        "ADC	r7, r7, r11\n\t"
        "STRD	r6, r7, [%[sha512], #40]\n\t"
        "MOV	r10, r8\n\t"
        "MOV	r11, r9\n\t"
        /* Round 3 */
        "LDRD	r4, r5, [%[sha512], #8]\n\t"
        "LSRS	r6, r4, #14\n\t"
        "LSRS	r7, r5, #14\n\t"
        "ORR	r7, r7, r4, LSL #18\n\t"
        "ORR	r6, r6, r5, LSL #18\n\t"
        "LSRS	r8, r4, #18\n\t"
        "LSRS	r9, r5, #18\n\t"
        "ORR	r9, r9, r4, LSL #14\n\t"
        "ORR	r8, r8, r5, LSL #14\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "LSLS	r8, r4, #23\n\t"
        "LSLS	r9, r5, #23\n\t"
        "ORR	r9, r9, r4, LSR #9\n\t"
        "ORR	r8, r8, r5, LSR #9\n\t"
        "LDRD	r4, r5, [%[sha512], #32]\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "STRD	r4, r5, [%[sha512], #32]\n\t"
        "LDRD	r4, r5, [%[sha512], #8]\n\t"
        "LDRD	r6, r7, [%[sha512], #16]\n\t"
        "LDRD	r8, r9, [%[sha512], #24]\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "AND	r6, r6, r4\n\t"
        "AND	r7, r7, r5\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "LDRD	r4, r5, [%[sha512], #32]\n\t"
        "LDRD	r8, r9, [sp, #24]\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "LDRD	r6, r7, [r3, #24]\n\t"
        "ADDS	r4, r4, r8\n\t"
        "ADC	r5, r5, r9\n\t"
        "LDRD	r8, r9, [%[sha512]]\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "STRD	r4, r5, [%[sha512], #32]\n\t"
        "ADDS	r8, r8, r4\n\t"
        "ADC	r9, r9, r5\n\t"
        "LDRD	r4, r5, [%[sha512], #40]\n\t"
        "STRD	r8, r9, [%[sha512]]\n\t"
        "LSRS	r6, r4, #28\n\t"
        "LSRS	r7, r5, #28\n\t"
        "ORR	r7, r7, r4, LSL #4\n\t"
        "ORR	r6, r6, r5, LSL #4\n\t"
        "LSLS	r8, r4, #30\n\t"
        "LSLS	r9, r5, #30\n\t"
        "ORR	r9, r9, r4, LSR #2\n\t"
        "ORR	r8, r8, r5, LSR #2\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "LSLS	r8, r4, #25\n\t"
        "LSLS	r9, r5, #25\n\t"
        "ORR	r9, r9, r4, LSR #7\n\t"
        "ORR	r8, r8, r5, LSR #7\n\t"
        "LDRD	r4, r5, [%[sha512], #32]\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "LDRD	r8, r9, [%[sha512], #40]\n\t"
        "LDRD	r6, r7, [%[sha512], #48]\n\t"
        "STRD	r4, r5, [%[sha512], #32]\n\t"
        "EOR	r8, r8, r6\n\t"
        "EOR	r9, r9, r7\n\t"
        "AND	r10, r10, r8\n\t"
        "AND	r11, r11, r9\n\t"
        "EOR	r10, r10, r6\n\t"
        "EOR	r11, r11, r7\n\t"
        "LDRD	r6, r7, [%[sha512], #32]\n\t"
        "ADDS	r6, r6, r10\n\t"
        "ADC	r7, r7, r11\n\t"
        "STRD	r6, r7, [%[sha512], #32]\n\t"
        "MOV	r10, r8\n\t"
        "MOV	r11, r9\n\t"
        /* Round 4 */
        "LDRD	r4, r5, [%[sha512]]\n\t"
        "LSRS	r6, r4, #14\n\t"
        "LSRS	r7, r5, #14\n\t"
        "ORR	r7, r7, r4, LSL #18\n\t"
        "ORR	r6, r6, r5, LSL #18\n\t"
        "LSRS	r8, r4, #18\n\t"
        "LSRS	r9, r5, #18\n\t"
        "ORR	r9, r9, r4, LSL #14\n\t"
        "ORR	r8, r8, r5, LSL #14\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "LSLS	r8, r4, #23\n\t"
        "LSLS	r9, r5, #23\n\t"
        "ORR	r9, r9, r4, LSR #9\n\t"
        "ORR	r8, r8, r5, LSR #9\n\t"
        "LDRD	r4, r5, [%[sha512], #24]\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "STRD	r4, r5, [%[sha512], #24]\n\t"
        "LDRD	r4, r5, [%[sha512]]\n\t"
        "LDRD	r6, r7, [%[sha512], #8]\n\t"
        "LDRD	r8, r9, [%[sha512], #16]\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "AND	r6, r6, r4\n\t"
        "AND	r7, r7, r5\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "LDRD	r4, r5, [%[sha512], #24]\n\t"
        "LDRD	r8, r9, [sp, #32]\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "LDRD	r6, r7, [r3, #32]\n\t"
        "ADDS	r4, r4, r8\n\t"
        "ADC	r5, r5, r9\n\t"
        "LDRD	r8, r9, [%[sha512], #56]\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "STRD	r4, r5, [%[sha512], #24]\n\t"
        "ADDS	r8, r8, r4\n\t"
        "ADC	r9, r9, r5\n\t"
        "LDRD	r4, r5, [%[sha512], #32]\n\t"
        "STRD	r8, r9, [%[sha512], #56]\n\t"
        "LSRS	r6, r4, #28\n\t"
        "LSRS	r7, r5, #28\n\t"
        "ORR	r7, r7, r4, LSL #4\n\t"
        "ORR	r6, r6, r5, LSL #4\n\t"
        "LSLS	r8, r4, #30\n\t"
        "LSLS	r9, r5, #30\n\t"
        "ORR	r9, r9, r4, LSR #2\n\t"
        "ORR	r8, r8, r5, LSR #2\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "LSLS	r8, r4, #25\n\t"
        "LSLS	r9, r5, #25\n\t"
        "ORR	r9, r9, r4, LSR #7\n\t"
        "ORR	r8, r8, r5, LSR #7\n\t"
        "LDRD	r4, r5, [%[sha512], #24]\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "LDRD	r8, r9, [%[sha512], #32]\n\t"
        "LDRD	r6, r7, [%[sha512], #40]\n\t"
        "STRD	r4, r5, [%[sha512], #24]\n\t"
        "EOR	r8, r8, r6\n\t"
        "EOR	r9, r9, r7\n\t"
        "AND	r10, r10, r8\n\t"
        "AND	r11, r11, r9\n\t"
        "EOR	r10, r10, r6\n\t"
        "EOR	r11, r11, r7\n\t"
        "LDRD	r6, r7, [%[sha512], #24]\n\t"
        "ADDS	r6, r6, r10\n\t"
        "ADC	r7, r7, r11\n\t"
        "STRD	r6, r7, [%[sha512], #24]\n\t"
        "MOV	r10, r8\n\t"
        "MOV	r11, r9\n\t"
        /* Round 5 */
        "LDRD	r4, r5, [%[sha512], #56]\n\t"
        "LSRS	r6, r4, #14\n\t"
        "LSRS	r7, r5, #14\n\t"
        "ORR	r7, r7, r4, LSL #18\n\t"
        "ORR	r6, r6, r5, LSL #18\n\t"
        "LSRS	r8, r4, #18\n\t"
        "LSRS	r9, r5, #18\n\t"
        "ORR	r9, r9, r4, LSL #14\n\t"
        "ORR	r8, r8, r5, LSL #14\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "LSLS	r8, r4, #23\n\t"
        "LSLS	r9, r5, #23\n\t"
        "ORR	r9, r9, r4, LSR #9\n\t"
        "ORR	r8, r8, r5, LSR #9\n\t"
        "LDRD	r4, r5, [%[sha512], #16]\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "STRD	r4, r5, [%[sha512], #16]\n\t"
        "LDRD	r4, r5, [%[sha512], #56]\n\t"
        "LDRD	r6, r7, [%[sha512]]\n\t"
        "LDRD	r8, r9, [%[sha512], #8]\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "AND	r6, r6, r4\n\t"
        "AND	r7, r7, r5\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "LDRD	r4, r5, [%[sha512], #16]\n\t"
        "LDRD	r8, r9, [sp, #40]\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "LDRD	r6, r7, [r3, #40]\n\t"
        "ADDS	r4, r4, r8\n\t"
        "ADC	r5, r5, r9\n\t"
        "LDRD	r8, r9, [%[sha512], #48]\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "STRD	r4, r5, [%[sha512], #16]\n\t"
        "ADDS	r8, r8, r4\n\t"
        "ADC	r9, r9, r5\n\t"
        "LDRD	r4, r5, [%[sha512], #24]\n\t"
        "STRD	r8, r9, [%[sha512], #48]\n\t"
        "LSRS	r6, r4, #28\n\t"
        "LSRS	r7, r5, #28\n\t"
        "ORR	r7, r7, r4, LSL #4\n\t"
        "ORR	r6, r6, r5, LSL #4\n\t"
        "LSLS	r8, r4, #30\n\t"
        "LSLS	r9, r5, #30\n\t"
        "ORR	r9, r9, r4, LSR #2\n\t"
        "ORR	r8, r8, r5, LSR #2\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "LSLS	r8, r4, #25\n\t"
        "LSLS	r9, r5, #25\n\t"
        "ORR	r9, r9, r4, LSR #7\n\t"
        "ORR	r8, r8, r5, LSR #7\n\t"
        "LDRD	r4, r5, [%[sha512], #16]\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "LDRD	r8, r9, [%[sha512], #24]\n\t"
        "LDRD	r6, r7, [%[sha512], #32]\n\t"
        "STRD	r4, r5, [%[sha512], #16]\n\t"
        "EOR	r8, r8, r6\n\t"
        "EOR	r9, r9, r7\n\t"
        "AND	r10, r10, r8\n\t"
        "AND	r11, r11, r9\n\t"
        "EOR	r10, r10, r6\n\t"
        "EOR	r11, r11, r7\n\t"
        "LDRD	r6, r7, [%[sha512], #16]\n\t"
        "ADDS	r6, r6, r10\n\t"
        "ADC	r7, r7, r11\n\t"
        "STRD	r6, r7, [%[sha512], #16]\n\t"
        "MOV	r10, r8\n\t"
        "MOV	r11, r9\n\t"
        /* Round 6 */
        "LDRD	r4, r5, [%[sha512], #48]\n\t"
        "LSRS	r6, r4, #14\n\t"
        "LSRS	r7, r5, #14\n\t"
        "ORR	r7, r7, r4, LSL #18\n\t"
        "ORR	r6, r6, r5, LSL #18\n\t"
        "LSRS	r8, r4, #18\n\t"
        "LSRS	r9, r5, #18\n\t"
        "ORR	r9, r9, r4, LSL #14\n\t"
        "ORR	r8, r8, r5, LSL #14\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "LSLS	r8, r4, #23\n\t"
        "LSLS	r9, r5, #23\n\t"
        "ORR	r9, r9, r4, LSR #9\n\t"
        "ORR	r8, r8, r5, LSR #9\n\t"
        "LDRD	r4, r5, [%[sha512], #8]\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "STRD	r4, r5, [%[sha512], #8]\n\t"
        "LDRD	r4, r5, [%[sha512], #48]\n\t"
        "LDRD	r6, r7, [%[sha512], #56]\n\t"
        "LDRD	r8, r9, [%[sha512]]\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "AND	r6, r6, r4\n\t"
        "AND	r7, r7, r5\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "LDRD	r4, r5, [%[sha512], #8]\n\t"
        "LDRD	r8, r9, [sp, #48]\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "LDRD	r6, r7, [r3, #48]\n\t"
        "ADDS	r4, r4, r8\n\t"
        "ADC	r5, r5, r9\n\t"
        "LDRD	r8, r9, [%[sha512], #40]\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "STRD	r4, r5, [%[sha512], #8]\n\t"
        "ADDS	r8, r8, r4\n\t"
        "ADC	r9, r9, r5\n\t"
        "LDRD	r4, r5, [%[sha512], #16]\n\t"
        "STRD	r8, r9, [%[sha512], #40]\n\t"
        "LSRS	r6, r4, #28\n\t"
        "LSRS	r7, r5, #28\n\t"
        "ORR	r7, r7, r4, LSL #4\n\t"
        "ORR	r6, r6, r5, LSL #4\n\t"
        "LSLS	r8, r4, #30\n\t"
        "LSLS	r9, r5, #30\n\t"
        "ORR	r9, r9, r4, LSR #2\n\t"
        "ORR	r8, r8, r5, LSR #2\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "LSLS	r8, r4, #25\n\t"
        "LSLS	r9, r5, #25\n\t"
        "ORR	r9, r9, r4, LSR #7\n\t"
        "ORR	r8, r8, r5, LSR #7\n\t"
        "LDRD	r4, r5, [%[sha512], #8]\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "LDRD	r8, r9, [%[sha512], #16]\n\t"
        "LDRD	r6, r7, [%[sha512], #24]\n\t"
        "STRD	r4, r5, [%[sha512], #8]\n\t"
        "EOR	r8, r8, r6\n\t"
        "EOR	r9, r9, r7\n\t"
        "AND	r10, r10, r8\n\t"
        "AND	r11, r11, r9\n\t"
        "EOR	r10, r10, r6\n\t"
        "EOR	r11, r11, r7\n\t"
        "LDRD	r6, r7, [%[sha512], #8]\n\t"
        "ADDS	r6, r6, r10\n\t"
        "ADC	r7, r7, r11\n\t"
        "STRD	r6, r7, [%[sha512], #8]\n\t"
        "MOV	r10, r8\n\t"
        "MOV	r11, r9\n\t"
        /* Round 7 */
        "LDRD	r4, r5, [%[sha512], #40]\n\t"
        "LSRS	r6, r4, #14\n\t"
        "LSRS	r7, r5, #14\n\t"
        "ORR	r7, r7, r4, LSL #18\n\t"
        "ORR	r6, r6, r5, LSL #18\n\t"
        "LSRS	r8, r4, #18\n\t"
        "LSRS	r9, r5, #18\n\t"
        "ORR	r9, r9, r4, LSL #14\n\t"
        "ORR	r8, r8, r5, LSL #14\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "LSLS	r8, r4, #23\n\t"
        "LSLS	r9, r5, #23\n\t"
        "ORR	r9, r9, r4, LSR #9\n\t"
        "ORR	r8, r8, r5, LSR #9\n\t"
        "LDRD	r4, r5, [%[sha512]]\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "STRD	r4, r5, [%[sha512]]\n\t"
        "LDRD	r4, r5, [%[sha512], #40]\n\t"
        "LDRD	r6, r7, [%[sha512], #48]\n\t"
        "LDRD	r8, r9, [%[sha512], #56]\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "AND	r6, r6, r4\n\t"
        "AND	r7, r7, r5\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "LDRD	r4, r5, [%[sha512]]\n\t"
        "LDRD	r8, r9, [sp, #56]\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "LDRD	r6, r7, [r3, #56]\n\t"
        "ADDS	r4, r4, r8\n\t"
        "ADC	r5, r5, r9\n\t"
        "LDRD	r8, r9, [%[sha512], #32]\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "STRD	r4, r5, [%[sha512]]\n\t"
        "ADDS	r8, r8, r4\n\t"
        "ADC	r9, r9, r5\n\t"
        "LDRD	r4, r5, [%[sha512], #8]\n\t"
        "STRD	r8, r9, [%[sha512], #32]\n\t"
        "LSRS	r6, r4, #28\n\t"
        "LSRS	r7, r5, #28\n\t"
        "ORR	r7, r7, r4, LSL #4\n\t"
        "ORR	r6, r6, r5, LSL #4\n\t"
        "LSLS	r8, r4, #30\n\t"
        "LSLS	r9, r5, #30\n\t"
        "ORR	r9, r9, r4, LSR #2\n\t"
        "ORR	r8, r8, r5, LSR #2\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "LSLS	r8, r4, #25\n\t"
        "LSLS	r9, r5, #25\n\t"
        "ORR	r9, r9, r4, LSR #7\n\t"
        "ORR	r8, r8, r5, LSR #7\n\t"
        "LDRD	r4, r5, [%[sha512]]\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "LDRD	r8, r9, [%[sha512], #8]\n\t"
        "LDRD	r6, r7, [%[sha512], #16]\n\t"
        "STRD	r4, r5, [%[sha512]]\n\t"
        "EOR	r8, r8, r6\n\t"
        "EOR	r9, r9, r7\n\t"
        "AND	r10, r10, r8\n\t"
        "AND	r11, r11, r9\n\t"
        "EOR	r10, r10, r6\n\t"
        "EOR	r11, r11, r7\n\t"
        "LDRD	r6, r7, [%[sha512]]\n\t"
        "ADDS	r6, r6, r10\n\t"
        "ADC	r7, r7, r11\n\t"
        "STRD	r6, r7, [%[sha512]]\n\t"
        "MOV	r10, r8\n\t"
        "MOV	r11, r9\n\t"
        /* Round 8 */
        "LDRD	r4, r5, [%[sha512], #32]\n\t"
        "LSRS	r6, r4, #14\n\t"
        "LSRS	r7, r5, #14\n\t"
        "ORR	r7, r7, r4, LSL #18\n\t"
        "ORR	r6, r6, r5, LSL #18\n\t"
        "LSRS	r8, r4, #18\n\t"
        "LSRS	r9, r5, #18\n\t"
        "ORR	r9, r9, r4, LSL #14\n\t"
        "ORR	r8, r8, r5, LSL #14\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "LSLS	r8, r4, #23\n\t"
        "LSLS	r9, r5, #23\n\t"
        "ORR	r9, r9, r4, LSR #9\n\t"
        "ORR	r8, r8, r5, LSR #9\n\t"
        "LDRD	r4, r5, [%[sha512], #56]\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "STRD	r4, r5, [%[sha512], #56]\n\t"
        "LDRD	r4, r5, [%[sha512], #32]\n\t"
        "LDRD	r6, r7, [%[sha512], #40]\n\t"
        "LDRD	r8, r9, [%[sha512], #48]\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "AND	r6, r6, r4\n\t"
        "AND	r7, r7, r5\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "LDRD	r4, r5, [%[sha512], #56]\n\t"
        "LDRD	r8, r9, [sp, #64]\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "LDRD	r6, r7, [r3, #64]\n\t"
        "ADDS	r4, r4, r8\n\t"
        "ADC	r5, r5, r9\n\t"
        "LDRD	r8, r9, [%[sha512], #24]\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "STRD	r4, r5, [%[sha512], #56]\n\t"
        "ADDS	r8, r8, r4\n\t"
        "ADC	r9, r9, r5\n\t"
        "LDRD	r4, r5, [%[sha512]]\n\t"
        "STRD	r8, r9, [%[sha512], #24]\n\t"
        "LSRS	r6, r4, #28\n\t"
        "LSRS	r7, r5, #28\n\t"
        "ORR	r7, r7, r4, LSL #4\n\t"
        "ORR	r6, r6, r5, LSL #4\n\t"
        "LSLS	r8, r4, #30\n\t"
        "LSLS	r9, r5, #30\n\t"
        "ORR	r9, r9, r4, LSR #2\n\t"
        "ORR	r8, r8, r5, LSR #2\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "LSLS	r8, r4, #25\n\t"
        "LSLS	r9, r5, #25\n\t"
        "ORR	r9, r9, r4, LSR #7\n\t"
        "ORR	r8, r8, r5, LSR #7\n\t"
        "LDRD	r4, r5, [%[sha512], #56]\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "LDRD	r8, r9, [%[sha512]]\n\t"
        "LDRD	r6, r7, [%[sha512], #8]\n\t"
        "STRD	r4, r5, [%[sha512], #56]\n\t"
        "EOR	r8, r8, r6\n\t"
        "EOR	r9, r9, r7\n\t"
        "AND	r10, r10, r8\n\t"
        "AND	r11, r11, r9\n\t"
        "EOR	r10, r10, r6\n\t"
        "EOR	r11, r11, r7\n\t"
        "LDRD	r6, r7, [%[sha512], #56]\n\t"
        "ADDS	r6, r6, r10\n\t"
        "ADC	r7, r7, r11\n\t"
        "STRD	r6, r7, [%[sha512], #56]\n\t"
        "MOV	r10, r8\n\t"
        "MOV	r11, r9\n\t"
        /* Round 9 */
        "LDRD	r4, r5, [%[sha512], #24]\n\t"
        "LSRS	r6, r4, #14\n\t"
        "LSRS	r7, r5, #14\n\t"
        "ORR	r7, r7, r4, LSL #18\n\t"
        "ORR	r6, r6, r5, LSL #18\n\t"
        "LSRS	r8, r4, #18\n\t"
        "LSRS	r9, r5, #18\n\t"
        "ORR	r9, r9, r4, LSL #14\n\t"
        "ORR	r8, r8, r5, LSL #14\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "LSLS	r8, r4, #23\n\t"
        "LSLS	r9, r5, #23\n\t"
        "ORR	r9, r9, r4, LSR #9\n\t"
        "ORR	r8, r8, r5, LSR #9\n\t"
        "LDRD	r4, r5, [%[sha512], #48]\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "STRD	r4, r5, [%[sha512], #48]\n\t"
        "LDRD	r4, r5, [%[sha512], #24]\n\t"
        "LDRD	r6, r7, [%[sha512], #32]\n\t"
        "LDRD	r8, r9, [%[sha512], #40]\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "AND	r6, r6, r4\n\t"
        "AND	r7, r7, r5\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "LDRD	r4, r5, [%[sha512], #48]\n\t"
        "LDRD	r8, r9, [sp, #72]\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "LDRD	r6, r7, [r3, #72]\n\t"
        "ADDS	r4, r4, r8\n\t"
        "ADC	r5, r5, r9\n\t"
        "LDRD	r8, r9, [%[sha512], #16]\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "STRD	r4, r5, [%[sha512], #48]\n\t"
        "ADDS	r8, r8, r4\n\t"
        "ADC	r9, r9, r5\n\t"
        "LDRD	r4, r5, [%[sha512], #56]\n\t"
        "STRD	r8, r9, [%[sha512], #16]\n\t"
        "LSRS	r6, r4, #28\n\t"
        "LSRS	r7, r5, #28\n\t"
        "ORR	r7, r7, r4, LSL #4\n\t"
        "ORR	r6, r6, r5, LSL #4\n\t"
        "LSLS	r8, r4, #30\n\t"
        "LSLS	r9, r5, #30\n\t"
        "ORR	r9, r9, r4, LSR #2\n\t"
        "ORR	r8, r8, r5, LSR #2\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "LSLS	r8, r4, #25\n\t"
        "LSLS	r9, r5, #25\n\t"
        "ORR	r9, r9, r4, LSR #7\n\t"
        "ORR	r8, r8, r5, LSR #7\n\t"
        "LDRD	r4, r5, [%[sha512], #48]\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "LDRD	r8, r9, [%[sha512], #56]\n\t"
        "LDRD	r6, r7, [%[sha512]]\n\t"
        "STRD	r4, r5, [%[sha512], #48]\n\t"
        "EOR	r8, r8, r6\n\t"
        "EOR	r9, r9, r7\n\t"
        "AND	r10, r10, r8\n\t"
        "AND	r11, r11, r9\n\t"
        "EOR	r10, r10, r6\n\t"
        "EOR	r11, r11, r7\n\t"
        "LDRD	r6, r7, [%[sha512], #48]\n\t"
        "ADDS	r6, r6, r10\n\t"
        "ADC	r7, r7, r11\n\t"
        "STRD	r6, r7, [%[sha512], #48]\n\t"
        "MOV	r10, r8\n\t"
        "MOV	r11, r9\n\t"
        /* Round 10 */
        "LDRD	r4, r5, [%[sha512], #16]\n\t"
        "LSRS	r6, r4, #14\n\t"
        "LSRS	r7, r5, #14\n\t"
        "ORR	r7, r7, r4, LSL #18\n\t"
        "ORR	r6, r6, r5, LSL #18\n\t"
        "LSRS	r8, r4, #18\n\t"
        "LSRS	r9, r5, #18\n\t"
        "ORR	r9, r9, r4, LSL #14\n\t"
        "ORR	r8, r8, r5, LSL #14\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "LSLS	r8, r4, #23\n\t"
        "LSLS	r9, r5, #23\n\t"
        "ORR	r9, r9, r4, LSR #9\n\t"
        "ORR	r8, r8, r5, LSR #9\n\t"
        "LDRD	r4, r5, [%[sha512], #40]\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "STRD	r4, r5, [%[sha512], #40]\n\t"
        "LDRD	r4, r5, [%[sha512], #16]\n\t"
        "LDRD	r6, r7, [%[sha512], #24]\n\t"
        "LDRD	r8, r9, [%[sha512], #32]\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "AND	r6, r6, r4\n\t"
        "AND	r7, r7, r5\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "LDRD	r4, r5, [%[sha512], #40]\n\t"
        "LDRD	r8, r9, [sp, #80]\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "LDRD	r6, r7, [r3, #80]\n\t"
        "ADDS	r4, r4, r8\n\t"
        "ADC	r5, r5, r9\n\t"
        "LDRD	r8, r9, [%[sha512], #8]\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "STRD	r4, r5, [%[sha512], #40]\n\t"
        "ADDS	r8, r8, r4\n\t"
        "ADC	r9, r9, r5\n\t"
        "LDRD	r4, r5, [%[sha512], #48]\n\t"
        "STRD	r8, r9, [%[sha512], #8]\n\t"
        "LSRS	r6, r4, #28\n\t"
        "LSRS	r7, r5, #28\n\t"
        "ORR	r7, r7, r4, LSL #4\n\t"
        "ORR	r6, r6, r5, LSL #4\n\t"
        "LSLS	r8, r4, #30\n\t"
        "LSLS	r9, r5, #30\n\t"
        "ORR	r9, r9, r4, LSR #2\n\t"
        "ORR	r8, r8, r5, LSR #2\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "LSLS	r8, r4, #25\n\t"
        "LSLS	r9, r5, #25\n\t"
        "ORR	r9, r9, r4, LSR #7\n\t"
        "ORR	r8, r8, r5, LSR #7\n\t"
        "LDRD	r4, r5, [%[sha512], #40]\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "LDRD	r8, r9, [%[sha512], #48]\n\t"
        "LDRD	r6, r7, [%[sha512], #56]\n\t"
        "STRD	r4, r5, [%[sha512], #40]\n\t"
        "EOR	r8, r8, r6\n\t"
        "EOR	r9, r9, r7\n\t"
        "AND	r10, r10, r8\n\t"
        "AND	r11, r11, r9\n\t"
        "EOR	r10, r10, r6\n\t"
        "EOR	r11, r11, r7\n\t"
        "LDRD	r6, r7, [%[sha512], #40]\n\t"
        "ADDS	r6, r6, r10\n\t"
        "ADC	r7, r7, r11\n\t"
        "STRD	r6, r7, [%[sha512], #40]\n\t"
        "MOV	r10, r8\n\t"
        "MOV	r11, r9\n\t"
        /* Round 11 */
        "LDRD	r4, r5, [%[sha512], #8]\n\t"
        "LSRS	r6, r4, #14\n\t"
        "LSRS	r7, r5, #14\n\t"
        "ORR	r7, r7, r4, LSL #18\n\t"
        "ORR	r6, r6, r5, LSL #18\n\t"
        "LSRS	r8, r4, #18\n\t"
        "LSRS	r9, r5, #18\n\t"
        "ORR	r9, r9, r4, LSL #14\n\t"
        "ORR	r8, r8, r5, LSL #14\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "LSLS	r8, r4, #23\n\t"
        "LSLS	r9, r5, #23\n\t"
        "ORR	r9, r9, r4, LSR #9\n\t"
        "ORR	r8, r8, r5, LSR #9\n\t"
        "LDRD	r4, r5, [%[sha512], #32]\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "STRD	r4, r5, [%[sha512], #32]\n\t"
        "LDRD	r4, r5, [%[sha512], #8]\n\t"
        "LDRD	r6, r7, [%[sha512], #16]\n\t"
        "LDRD	r8, r9, [%[sha512], #24]\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "AND	r6, r6, r4\n\t"
        "AND	r7, r7, r5\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "LDRD	r4, r5, [%[sha512], #32]\n\t"
        "LDRD	r8, r9, [sp, #88]\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "LDRD	r6, r7, [r3, #88]\n\t"
        "ADDS	r4, r4, r8\n\t"
        "ADC	r5, r5, r9\n\t"
        "LDRD	r8, r9, [%[sha512]]\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "STRD	r4, r5, [%[sha512], #32]\n\t"
        "ADDS	r8, r8, r4\n\t"
        "ADC	r9, r9, r5\n\t"
        "LDRD	r4, r5, [%[sha512], #40]\n\t"
        "STRD	r8, r9, [%[sha512]]\n\t"
        "LSRS	r6, r4, #28\n\t"
        "LSRS	r7, r5, #28\n\t"
        "ORR	r7, r7, r4, LSL #4\n\t"
        "ORR	r6, r6, r5, LSL #4\n\t"
        "LSLS	r8, r4, #30\n\t"
        "LSLS	r9, r5, #30\n\t"
        "ORR	r9, r9, r4, LSR #2\n\t"
        "ORR	r8, r8, r5, LSR #2\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "LSLS	r8, r4, #25\n\t"
        "LSLS	r9, r5, #25\n\t"
        "ORR	r9, r9, r4, LSR #7\n\t"
        "ORR	r8, r8, r5, LSR #7\n\t"
        "LDRD	r4, r5, [%[sha512], #32]\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "LDRD	r8, r9, [%[sha512], #40]\n\t"
        "LDRD	r6, r7, [%[sha512], #48]\n\t"
        "STRD	r4, r5, [%[sha512], #32]\n\t"
        "EOR	r8, r8, r6\n\t"
        "EOR	r9, r9, r7\n\t"
        "AND	r10, r10, r8\n\t"
        "AND	r11, r11, r9\n\t"
        "EOR	r10, r10, r6\n\t"
        "EOR	r11, r11, r7\n\t"
        "LDRD	r6, r7, [%[sha512], #32]\n\t"
        "ADDS	r6, r6, r10\n\t"
        "ADC	r7, r7, r11\n\t"
        "STRD	r6, r7, [%[sha512], #32]\n\t"
        "MOV	r10, r8\n\t"
        "MOV	r11, r9\n\t"
        /* Round 12 */
        "LDRD	r4, r5, [%[sha512]]\n\t"
        "LSRS	r6, r4, #14\n\t"
        "LSRS	r7, r5, #14\n\t"
        "ORR	r7, r7, r4, LSL #18\n\t"
        "ORR	r6, r6, r5, LSL #18\n\t"
        "LSRS	r8, r4, #18\n\t"
        "LSRS	r9, r5, #18\n\t"
        "ORR	r9, r9, r4, LSL #14\n\t"
        "ORR	r8, r8, r5, LSL #14\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "LSLS	r8, r4, #23\n\t"
        "LSLS	r9, r5, #23\n\t"
        "ORR	r9, r9, r4, LSR #9\n\t"
        "ORR	r8, r8, r5, LSR #9\n\t"
        "LDRD	r4, r5, [%[sha512], #24]\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "STRD	r4, r5, [%[sha512], #24]\n\t"
        "LDRD	r4, r5, [%[sha512]]\n\t"
        "LDRD	r6, r7, [%[sha512], #8]\n\t"
        "LDRD	r8, r9, [%[sha512], #16]\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "AND	r6, r6, r4\n\t"
        "AND	r7, r7, r5\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "LDRD	r4, r5, [%[sha512], #24]\n\t"
        "LDRD	r8, r9, [sp, #96]\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "LDRD	r6, r7, [r3, #96]\n\t"
        "ADDS	r4, r4, r8\n\t"
        "ADC	r5, r5, r9\n\t"
        "LDRD	r8, r9, [%[sha512], #56]\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "STRD	r4, r5, [%[sha512], #24]\n\t"
        "ADDS	r8, r8, r4\n\t"
        "ADC	r9, r9, r5\n\t"
        "LDRD	r4, r5, [%[sha512], #32]\n\t"
        "STRD	r8, r9, [%[sha512], #56]\n\t"
        "LSRS	r6, r4, #28\n\t"
        "LSRS	r7, r5, #28\n\t"
        "ORR	r7, r7, r4, LSL #4\n\t"
        "ORR	r6, r6, r5, LSL #4\n\t"
        "LSLS	r8, r4, #30\n\t"
        "LSLS	r9, r5, #30\n\t"
        "ORR	r9, r9, r4, LSR #2\n\t"
        "ORR	r8, r8, r5, LSR #2\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "LSLS	r8, r4, #25\n\t"
        "LSLS	r9, r5, #25\n\t"
        "ORR	r9, r9, r4, LSR #7\n\t"
        "ORR	r8, r8, r5, LSR #7\n\t"
        "LDRD	r4, r5, [%[sha512], #24]\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "LDRD	r8, r9, [%[sha512], #32]\n\t"
        "LDRD	r6, r7, [%[sha512], #40]\n\t"
        "STRD	r4, r5, [%[sha512], #24]\n\t"
        "EOR	r8, r8, r6\n\t"
        "EOR	r9, r9, r7\n\t"
        "AND	r10, r10, r8\n\t"
        "AND	r11, r11, r9\n\t"
        "EOR	r10, r10, r6\n\t"
        "EOR	r11, r11, r7\n\t"
        "LDRD	r6, r7, [%[sha512], #24]\n\t"
        "ADDS	r6, r6, r10\n\t"
        "ADC	r7, r7, r11\n\t"
        "STRD	r6, r7, [%[sha512], #24]\n\t"
        "MOV	r10, r8\n\t"
        "MOV	r11, r9\n\t"
        /* Round 13 */
        "LDRD	r4, r5, [%[sha512], #56]\n\t"
        "LSRS	r6, r4, #14\n\t"
        "LSRS	r7, r5, #14\n\t"
        "ORR	r7, r7, r4, LSL #18\n\t"
        "ORR	r6, r6, r5, LSL #18\n\t"
        "LSRS	r8, r4, #18\n\t"
        "LSRS	r9, r5, #18\n\t"
        "ORR	r9, r9, r4, LSL #14\n\t"
        "ORR	r8, r8, r5, LSL #14\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "LSLS	r8, r4, #23\n\t"
        "LSLS	r9, r5, #23\n\t"
        "ORR	r9, r9, r4, LSR #9\n\t"
        "ORR	r8, r8, r5, LSR #9\n\t"
        "LDRD	r4, r5, [%[sha512], #16]\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "STRD	r4, r5, [%[sha512], #16]\n\t"
        "LDRD	r4, r5, [%[sha512], #56]\n\t"
        "LDRD	r6, r7, [%[sha512]]\n\t"
        "LDRD	r8, r9, [%[sha512], #8]\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "AND	r6, r6, r4\n\t"
        "AND	r7, r7, r5\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "LDRD	r4, r5, [%[sha512], #16]\n\t"
        "LDRD	r8, r9, [sp, #104]\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "LDRD	r6, r7, [r3, #104]\n\t"
        "ADDS	r4, r4, r8\n\t"
        "ADC	r5, r5, r9\n\t"
        "LDRD	r8, r9, [%[sha512], #48]\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "STRD	r4, r5, [%[sha512], #16]\n\t"
        "ADDS	r8, r8, r4\n\t"
        "ADC	r9, r9, r5\n\t"
        "LDRD	r4, r5, [%[sha512], #24]\n\t"
        "STRD	r8, r9, [%[sha512], #48]\n\t"
        "LSRS	r6, r4, #28\n\t"
        "LSRS	r7, r5, #28\n\t"
        "ORR	r7, r7, r4, LSL #4\n\t"
        "ORR	r6, r6, r5, LSL #4\n\t"
        "LSLS	r8, r4, #30\n\t"
        "LSLS	r9, r5, #30\n\t"
        "ORR	r9, r9, r4, LSR #2\n\t"
        "ORR	r8, r8, r5, LSR #2\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "LSLS	r8, r4, #25\n\t"
        "LSLS	r9, r5, #25\n\t"
        "ORR	r9, r9, r4, LSR #7\n\t"
        "ORR	r8, r8, r5, LSR #7\n\t"
        "LDRD	r4, r5, [%[sha512], #16]\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "LDRD	r8, r9, [%[sha512], #24]\n\t"
        "LDRD	r6, r7, [%[sha512], #32]\n\t"
        "STRD	r4, r5, [%[sha512], #16]\n\t"
        "EOR	r8, r8, r6\n\t"
        "EOR	r9, r9, r7\n\t"
        "AND	r10, r10, r8\n\t"
        "AND	r11, r11, r9\n\t"
        "EOR	r10, r10, r6\n\t"
        "EOR	r11, r11, r7\n\t"
        "LDRD	r6, r7, [%[sha512], #16]\n\t"
        "ADDS	r6, r6, r10\n\t"
        "ADC	r7, r7, r11\n\t"
        "STRD	r6, r7, [%[sha512], #16]\n\t"
        "MOV	r10, r8\n\t"
        "MOV	r11, r9\n\t"
        /* Round 14 */
        "LDRD	r4, r5, [%[sha512], #48]\n\t"
        "LSRS	r6, r4, #14\n\t"
        "LSRS	r7, r5, #14\n\t"
        "ORR	r7, r7, r4, LSL #18\n\t"
        "ORR	r6, r6, r5, LSL #18\n\t"
        "LSRS	r8, r4, #18\n\t"
        "LSRS	r9, r5, #18\n\t"
        "ORR	r9, r9, r4, LSL #14\n\t"
        "ORR	r8, r8, r5, LSL #14\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "LSLS	r8, r4, #23\n\t"
        "LSLS	r9, r5, #23\n\t"
        "ORR	r9, r9, r4, LSR #9\n\t"
        "ORR	r8, r8, r5, LSR #9\n\t"
        "LDRD	r4, r5, [%[sha512], #8]\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "STRD	r4, r5, [%[sha512], #8]\n\t"
        "LDRD	r4, r5, [%[sha512], #48]\n\t"
        "LDRD	r6, r7, [%[sha512], #56]\n\t"
        "LDRD	r8, r9, [%[sha512]]\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "AND	r6, r6, r4\n\t"
        "AND	r7, r7, r5\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "LDRD	r4, r5, [%[sha512], #8]\n\t"
        "LDRD	r8, r9, [sp, #112]\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "LDRD	r6, r7, [r3, #112]\n\t"
        "ADDS	r4, r4, r8\n\t"
        "ADC	r5, r5, r9\n\t"
        "LDRD	r8, r9, [%[sha512], #40]\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "STRD	r4, r5, [%[sha512], #8]\n\t"
        "ADDS	r8, r8, r4\n\t"
        "ADC	r9, r9, r5\n\t"
        "LDRD	r4, r5, [%[sha512], #16]\n\t"
        "STRD	r8, r9, [%[sha512], #40]\n\t"
        "LSRS	r6, r4, #28\n\t"
        "LSRS	r7, r5, #28\n\t"
        "ORR	r7, r7, r4, LSL #4\n\t"
        "ORR	r6, r6, r5, LSL #4\n\t"
        "LSLS	r8, r4, #30\n\t"
        "LSLS	r9, r5, #30\n\t"
        "ORR	r9, r9, r4, LSR #2\n\t"
        "ORR	r8, r8, r5, LSR #2\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "LSLS	r8, r4, #25\n\t"
        "LSLS	r9, r5, #25\n\t"
        "ORR	r9, r9, r4, LSR #7\n\t"
        "ORR	r8, r8, r5, LSR #7\n\t"
        "LDRD	r4, r5, [%[sha512], #8]\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "LDRD	r8, r9, [%[sha512], #16]\n\t"
        "LDRD	r6, r7, [%[sha512], #24]\n\t"
        "STRD	r4, r5, [%[sha512], #8]\n\t"
        "EOR	r8, r8, r6\n\t"
        "EOR	r9, r9, r7\n\t"
        "AND	r10, r10, r8\n\t"
        "AND	r11, r11, r9\n\t"
        "EOR	r10, r10, r6\n\t"
        "EOR	r11, r11, r7\n\t"
        "LDRD	r6, r7, [%[sha512], #8]\n\t"
        "ADDS	r6, r6, r10\n\t"
        "ADC	r7, r7, r11\n\t"
        "STRD	r6, r7, [%[sha512], #8]\n\t"
        "MOV	r10, r8\n\t"
        "MOV	r11, r9\n\t"
        /* Round 15 */
        "LDRD	r4, r5, [%[sha512], #40]\n\t"
        "LSRS	r6, r4, #14\n\t"
        "LSRS	r7, r5, #14\n\t"
        "ORR	r7, r7, r4, LSL #18\n\t"
        "ORR	r6, r6, r5, LSL #18\n\t"
        "LSRS	r8, r4, #18\n\t"
        "LSRS	r9, r5, #18\n\t"
        "ORR	r9, r9, r4, LSL #14\n\t"
        "ORR	r8, r8, r5, LSL #14\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "LSLS	r8, r4, #23\n\t"
        "LSLS	r9, r5, #23\n\t"
        "ORR	r9, r9, r4, LSR #9\n\t"
        "ORR	r8, r8, r5, LSR #9\n\t"
        "LDRD	r4, r5, [%[sha512]]\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "STRD	r4, r5, [%[sha512]]\n\t"
        "LDRD	r4, r5, [%[sha512], #40]\n\t"
        "LDRD	r6, r7, [%[sha512], #48]\n\t"
        "LDRD	r8, r9, [%[sha512], #56]\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "AND	r6, r6, r4\n\t"
        "AND	r7, r7, r5\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "LDRD	r4, r5, [%[sha512]]\n\t"
        "LDRD	r8, r9, [sp, #120]\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "LDRD	r6, r7, [r3, #120]\n\t"
        "ADDS	r4, r4, r8\n\t"
        "ADC	r5, r5, r9\n\t"
        "LDRD	r8, r9, [%[sha512], #32]\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "STRD	r4, r5, [%[sha512]]\n\t"
        "ADDS	r8, r8, r4\n\t"
        "ADC	r9, r9, r5\n\t"
        "LDRD	r4, r5, [%[sha512], #8]\n\t"
        "STRD	r8, r9, [%[sha512], #32]\n\t"
        "LSRS	r6, r4, #28\n\t"
        "LSRS	r7, r5, #28\n\t"
        "ORR	r7, r7, r4, LSL #4\n\t"
        "ORR	r6, r6, r5, LSL #4\n\t"
        "LSLS	r8, r4, #30\n\t"
        "LSLS	r9, r5, #30\n\t"
        "ORR	r9, r9, r4, LSR #2\n\t"
        "ORR	r8, r8, r5, LSR #2\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "LSLS	r8, r4, #25\n\t"
        "LSLS	r9, r5, #25\n\t"
        "ORR	r9, r9, r4, LSR #7\n\t"
        "ORR	r8, r8, r5, LSR #7\n\t"
        "LDRD	r4, r5, [%[sha512]]\n\t"
        "EOR	r6, r6, r8\n\t"
        "EOR	r7, r7, r9\n\t"
        "ADDS	r4, r4, r6\n\t"
        "ADC	r5, r5, r7\n\t"
        "LDRD	r8, r9, [%[sha512], #8]\n\t"
        "LDRD	r6, r7, [%[sha512], #16]\n\t"
        "STRD	r4, r5, [%[sha512]]\n\t"
        "EOR	r8, r8, r6\n\t"
        "EOR	r9, r9, r7\n\t"
        "AND	r10, r10, r8\n\t"
        "AND	r11, r11, r9\n\t"
        "EOR	r10, r10, r6\n\t"
        "EOR	r11, r11, r7\n\t"
        "LDRD	r6, r7, [%[sha512]]\n\t"
        "ADDS	r6, r6, r10\n\t"
        "ADC	r7, r7, r11\n\t"
        "STRD	r6, r7, [%[sha512]]\n\t"
        "MOV	r10, r8\n\t"
        "MOV	r11, r9\n\t"
        /* Add in digest from start */
        "LDRD	r4, r5, [%[sha512]]\n\t"
        "LDRD	r6, r7, [%[sha512], #8]\n\t"
        "LDRD	r8, r9, [sp, #128]\n\t"
        "LDRD	r10, r11, [sp, #136]\n\t"
        "ADDS	r4, r4, r8\n\t"
        "ADC	r5, r5, r9\n\t"
        "ADDS	r6, r6, r10\n\t"
        "ADC	r7, r7, r11\n\t"
        "STRD	r4, r5, [%[sha512]]\n\t"
        "STRD	r6, r7, [%[sha512], #8]\n\t"
        "STRD	r4, r5, [sp, #128]\n\t"
        "STRD	r6, r7, [sp, #136]\n\t"
        "LDRD	r4, r5, [%[sha512], #16]\n\t"
        "LDRD	r6, r7, [%[sha512], #24]\n\t"
        "LDRD	r8, r9, [sp, #144]\n\t"
        "LDRD	r10, r11, [sp, #152]\n\t"
        "ADDS	r4, r4, r8\n\t"
        "ADC	r5, r5, r9\n\t"
        "ADDS	r6, r6, r10\n\t"
        "ADC	r7, r7, r11\n\t"
        "STRD	r4, r5, [%[sha512], #16]\n\t"
        "STRD	r6, r7, [%[sha512], #24]\n\t"
        "STRD	r4, r5, [sp, #144]\n\t"
        "STRD	r6, r7, [sp, #152]\n\t"
        "LDRD	r4, r5, [%[sha512], #32]\n\t"
        "LDRD	r6, r7, [%[sha512], #40]\n\t"
        "LDRD	r8, r9, [sp, #160]\n\t"
        "LDRD	r10, r11, [sp, #168]\n\t"
        "ADDS	r4, r4, r8\n\t"
        "ADC	r5, r5, r9\n\t"
        "ADDS	r6, r6, r10\n\t"
        "ADC	r7, r7, r11\n\t"
        "STRD	r4, r5, [%[sha512], #32]\n\t"
        "STRD	r6, r7, [%[sha512], #40]\n\t"
        "STRD	r4, r5, [sp, #160]\n\t"
        "STRD	r6, r7, [sp, #168]\n\t"
        "LDRD	r4, r5, [%[sha512], #48]\n\t"
        "LDRD	r6, r7, [%[sha512], #56]\n\t"
        "LDRD	r8, r9, [sp, #176]\n\t"
        "LDRD	r10, r11, [sp, #184]\n\t"
        "ADDS	r4, r4, r8\n\t"
        "ADC	r5, r5, r9\n\t"
        "ADDS	r6, r6, r10\n\t"
        "ADC	r7, r7, r11\n\t"
        "STRD	r4, r5, [%[sha512], #48]\n\t"
        "STRD	r6, r7, [%[sha512], #56]\n\t"
        "STRD	r4, r5, [sp, #176]\n\t"
        "STRD	r6, r7, [sp, #184]\n\t"
        "SUBS	%[len], %[len], #0x80\n\t"
        "SUB	r3, r3, #0x200\n\t"
        "ADD	%[data], %[data], #0x80\n\t"
#ifdef __GNUC__
        "BNE	L_SHA512_transform_len_begin%=\n\t"
#else
        "BNE.W	L_SHA512_transform_len_begin%=\n\t"
#endif
        "EOR	r0, r0, r0\n\t"
        "ADD	sp, sp, #0xc0\n\t"
#ifndef WOLFSSL_NO_VAR_ASSIGN_REG
        : [sha512] "+r" (sha512), [data] "+r" (data), [len] "+r" (len),
          [L_SHA512_transform_len_k] "+r" (L_SHA512_transform_len_k_c)
        :
        : "memory", "r4", "r5", "r6", "r7", "r8", "r9", "r10", "r11", "r12", "cc"
#else
        : [sha512] "+r" (sha512), [data] "+r" (data), [len] "+r" (len)
        : [L_SHA512_transform_len_k] "r" (L_SHA512_transform_len_k)
        : "memory", "r4", "r5", "r6", "r7", "r8", "r9", "r10", "r11", "r12", "cc"
#endif /* WOLFSSL_NO_VAR_ASSIGN_REG */
    );
}

#endif /* WOLFSSL_ARMASM_NO_NEON */
#endif /* WOLFSSL_SHA512 */
#endif /* !__aarch64__ && __thumb__ */
#endif /* WOLFSSL_ARMASM */
#endif /* WOLFSSL_ARMASM_INLINE */
