// Copyright (c) .NET Foundation. All rights reserved.
// Licensed under the Apache License, Version 2.0. See License.txt in the project root for license information.

#nullable enable

using System;
using System.Collections.Generic;
using System.Threading;
using System.Threading.Tasks;
using FluentAssertions;
using NuGet.Configuration;
using NuGet.Protocol.Core.Types;
using NuGet.Protocol.Model;
using NuGet.Protocol.Resources;
using NuGet.Protocol.Tests.Providers;
using NuGet.Test.Utility;
using Test.Utility;
using Xunit;

namespace NuGet.Protocol.Tests.Resources
{
    public class VulnerabilityInfoResourceV3Tests
    {
        [Fact]
        public async Task GetVulnerabilityFilesAsync_FeedHasVulnerabilityIndex_ReturnsVulnerabilityIndexEntries()
        {
            // Arrange
            string serviceIndexUrl = "https://nuget.test/v3/index.json";
            string vulnerabilityInfoIndexUrl = "https://nuget.test/v3/vulnerabilities/index.json";
            string vulnerabilityInfoFile1Url = "https://nuget.test/v3/vulnerabilities/base.json";
            string vulnerabilityInfoFile1Updated = "2023-03-14T14:15:16Z";
            string vulnerabilityInfoFile2Url = "https://nuget.test/v3/vulnerabilities/update.json";
            string vulnerabilityInfoFile2Updated = "2023-03-14T23:22:21Z";


            var serviceIndexProvider = MockServiceIndexResourceV3Provider.Create(
                new ServiceIndexEntry(new Uri(vulnerabilityInfoIndexUrl), ServiceTypes.VulnerabilityInfo[0], new Versioning.SemanticVersion(1, 0, 0)));

            string vulnerabilityInfoIndexContent = $@"[
    {{
        ""@name"": ""base"",
        ""@id"": ""{vulnerabilityInfoFile1Url}"",
        ""@updated"": ""{vulnerabilityInfoFile1Updated}""
    }},
    {{
        ""@name"": ""update"",
        ""@id"": ""{vulnerabilityInfoFile2Url}"",
        ""@updated"": ""{vulnerabilityInfoFile2Updated}""
    }}
]";

            var responses = new Dictionary<string, string>()
            {
                [vulnerabilityInfoIndexUrl] = vulnerabilityInfoIndexContent
            };
            var packageSource = new PackageSource(serviceIndexUrl);

            var sourceRepository = new SourceRepository(packageSource, new INuGetResourceProvider[]
            {
                StaticHttpSource.CreateHttpSource(responses),
                serviceIndexProvider
            });

            VulnerabilityInfoResourceV3 target = new(sourceRepository);

            SourceCacheContext context = new();
            TestLogger logger = new();

            // Act
            var files = await target.GetVulnerabilityFilesAsync(context, logger, CancellationToken.None);

            // Assert
            files.Should().NotBeNull();
            files.Count.Should().Be(2);

            files[0].Name.Should().Be("base");
            files[0].Url.Should().Be(vulnerabilityInfoFile1Url);
            files[0].Updated.Should().Be(vulnerabilityInfoFile1Updated);

            files[1].Name.Should().Be("update");
            files[1].Url.Should().Be(vulnerabilityInfoFile2Url);
            files[1].Updated.Should().Be(vulnerabilityInfoFile2Updated);
        }

        [Fact]
        public async Task GetVulnerabilityDataAsync_UrlExists_ReturnsVulnerabilityData()
        {
            // Arrange
            string serviceIndexUrl = "https://nuget.test/v3/index.json";
            string vulnerabilityDataUrl = "https://nuget.test/v3/vulnerabilities/base.json";

            string packageId = "packagea";
            string cveUrl = "https://contoso.test/advisories/CVE-ABC123";
            string rangeString = "(, 1.0.0]";

            string vulnerabilityDataContent = $@"{{
    ""{packageId}"": [
        {{
            ""url"": ""{cveUrl}"",
            ""severity"": 1,
            ""versions"": ""{rangeString}""
        }}
    ]
}}";

            var responses = new Dictionary<string, string>()
            {
                [vulnerabilityDataUrl] = vulnerabilityDataContent
            };
            var packageSource = new PackageSource(serviceIndexUrl);

            var sourceRepository = new SourceRepository(packageSource, new INuGetResourceProvider[]
            {
                StaticHttpSource.CreateHttpSource(responses)
            });

            VulnerabilityInfoResourceV3 target = new(sourceRepository);

            SourceCacheContext context = new();
            TestLogger logger = new();

            V3VulnerabilityIndexEntry vulnerabilityPage = new("base", new Uri(vulnerabilityDataUrl), "1234", comment: null);

            // Act
            IReadOnlyDictionary<string, IReadOnlyList<PackageVulnerabilityInfo>> data =
                await target.GetVulnerabilityDataAsync(vulnerabilityPage, context, logger, CancellationToken.None);

            // Assert
            data.Should().NotBeNull();
            data.Count.Should().Be(1);
            data.ContainsKey(packageId).Should().BeTrue();

            IReadOnlyList<PackageVulnerabilityInfo> packageVulnerability = data[packageId];
            packageVulnerability.Should().NotBeNull();
            packageVulnerability.Count.Should().Be(1);
            packageVulnerability[0].Url.OriginalString.Should().Be(cveUrl);
            packageVulnerability[0].Severity.Should().Be(PackageVulnerabilitySeverity.Moderate);
            packageVulnerability[0].Versions.OriginalString.Should().Be(rangeString);
        }
    }
}
